#!/bin/sh

########################################################################
#             (c) Copyright 2002 Lexmark International, Inc.           #
#                        All rights reserved                           #
########################################################################
#                                                                       
# This command is used to verify virtual device names, physical devices
# and/or print server settings.  
#
########################################################################
#
# Exit codes:
# 0   --- Success - Internal Print Server -- Valid character device
# 1   --- Success - External Print Server - 1 Port
# 2   --- Success - External Print Server - 2 Port
# 3   --- Success - External Print Server - 3 Port
# 4   --- Success - Unable to contact device [ Down or Unknown ]
#
#  9  --- Error   - Invalid Device Name
# 10  --- Error   - Device Name already exists
# 11  --- Error   - Invalid Device File
# 12  --- Error   - Invalid User ID
#
# 13  --- Error   - Device Name already exists & Invalid Device File
# 14  --- Error   - Device Name already exists & Invalid user ID
# 15  --- Error   - Invalid Device File & Invalid user ID
# 16  --- Error   - Device Name already exists & Invalid Device File & Invalid user ID
#
# 17  --- Error   - Device Name already exists & Unknown print server type
# 18  --- Error   - Device Name already exists & Unknown print server type & Invalid user ID
# 19  --- Error   - Unknown print server type & Invalid user ID
#
# 20  --- Error   - Device Name already exists & Invalid IP Address or Hostname
# 21  --- Error   - Device Name already exists & Invalid IP Address or Hostname & Invalid user ID 
# 22  --- Error   - Invalid IP Address or Hostname & Invalid user ID
#
# 101 --- Error   - Invalid Hostname or IP address
# 104 --- Error   - Invalid link-local IPv6 address
# 
#  99 --- Error   - Command line options incorrect
#
########################################################################

#-- set debug to false by default. To turn debug "on" set to "1"
DEBUG=0

#-- set initial return_code
return_code=99

#-- initialize variables to 0
lsdev_return=0
type_return=0
user_return=0


PRODUCT_LINK="/usr/genprint"
SCRIPTS_DIR="${PRODUCT_LINK}/bin/.scripts"

#-- load the shared script
. ${SCRIPTS_DIR}/shared

#-- initialize
initialize

program_name=`${BASENAME_CMD} $0`

#-- usage_short text
usage_short ()
{
${CAT_CMD} << EOT

USAGE:
    ${program_name} -d device [-x]
    ${program_name} -u username [-x]
    ${program_name} -i host [-c name] [-d device] [-u username] [-x]
    ${program_name} -p file [-d device] [-u username] [-x]
    ${program_name} -h
    ${program_name} -v

Try '${program_name} -h' for more information

EOT
}

#-- usage_long text
usage_long ()
{
${CAT_CMD} << EOT

${program_name}
    Verifies that the virtual device name is available and all local and/or
    network options are valid.
        
USAGE:
    ${program_name} -d device [-x]
    ${program_name} -u username [-x]
    ${program_name} -i host [-c name] [-d device] [-u username] [-x]
    ${program_name} -p file [-d device] [-u username] [-x]
    ${program_name} -h
    ${program_name} -v

OPTIONS:
    -d device
       Specify the virtual device name.

    -u username
       Specify a valid username.

    -i host
       Specify a hostname or IP address.
        
    -c name
       Specify the SNMP community name.  The default value is 'public'.

    -p filename
       Specify the absolute device filename.

    -x
       Run ${program_name} in debug mode.

    -h
       Display this help and exit.

    -v
       Display version information and exit.

EXAMPLES:
    Check that the device name mydevice is available:

       $ ${program_name} -d mydevice

    Check that the device name mydevice is available and the device /dev/lp0 is
    available:

       $ ${program_name} -p /dev/lp0 -d mydevice

    Check that the device name mydevice is available and the network options on
    host somehost are valid using the SNMP community name mystring:

       $ ${program_name} -i somehost -c mystring -d mydevice

EOT
}

#---------------------------------------------------
#--  This function is called to check for a physical 
#--  character device.
#--
#--  Syntax:
#--  check_physical_device <device>
#--
#--  Return 0 : Device exists
#--  Return 1 : Device does not exists
#--
#--
check_physical_devices()
{
   ldevice_file="${1}"
 
   # Check for Character Device
   if [ -c "${ldevice_file}" ];then 
      pdebug "Character Device [ ${ldevice_file} ] is valid\n"
      pdebug "Return Code: 0\n"
      return_code=0
   else
      pdebug "Character Device [ ${ldevice_file} ] is not valid\n"
      pdebug "Return Code: 1\n"
      return_code=3
   fi
   return ${return_code}
}

#---------------------------------------------------
#--  This function is called to check for network
#--  devices
#--
#--  Syntax:
#--  check_network_device <hostname> <community_name>
#--
#--  Return 0 : Device exists
#--  Return 1 : Device does not exists
#--
#--  Return 101 : Invalid Hostname or IP address
#--  Return 102 : Invalid IP address 
#--  Return 103 : Invalid IP address range
#--  Return 104 : Invalid link-local IPv6 address
#--
check_network_device()
{
lhost="${1}"
lcname="${2}"
lusesearchv6="${3}"

if [ "${DEBUG}" = "0" ]; then
   NPS_FILE="/dev/null"
else
   NPS_FILE="/tmp/printer_search.$$"
fi

NPSEARCH_CMD=printer_search6
DELIM="#"
${PRODUCT_LINK}/bin/${NPSEARCH_CMD} -q -h "${lhost}" "${NPS_FILE}" 2>/dev/null

if [ ! ${?} -eq 31 ];then
	if [ "${lusesearchv6}" = "TRUE" ];then
		NPSEARCH_CMD="printer_search6 -4"
		DELIM="#"
	else
		NPSEARCH_CMD=printer_search
        	DELIM=":"
	fi
fi

${PRODUCT_LINK}/bin/${NPSEARCH_CMD} -h "${lhost}" -c "${lcname}","${lcname}" -x "${NPS_FILE}" 2>&1 | \
      ${AWK_CMD} -F"${DELIM}" '
	  BEGIN { num_of_ports=10 }  # return unknown if no devices found
          /Invalid Hostname or IP address/  { num_of_ports=101 ; exit  }
          /Invalid IP address:/             { num_of_ports=102 ; exit  }
          /IPv4 address/                    { num_of_ports=102 ; exit  }
          /Invalid IP address range/        { num_of_ports=103 ; exit  }
          /Invalid link-local/              { num_of_ports=104 ; exit  }
          {

          if ( NR == 1 ) {
             outline = searched_host "#" $4 "#" $3
          } else {
             outline = outline "#" $3
          }

          num_of_ports = $6

          }
          END {
                if ( num_of_ports == "3" ) outline = outline ":Unknown Printer"
                if ( num_of_ports == "4" ) num_of_ports = 44
                if ( num_of_ports == "10" ) num_of_ports = 4

                if ( num_of_ports > 100 && num_of_ports < 105 ) print
                else if ( num_of_ports != "4" ) printf("%s\n", outline)
                else if ( num_of_ports = "4" ) printf("%s\n", searched_host)

                exit num_of_ports
           }' searched_host="${lhost}"
   return ${?}
}


#-- MAIN --#
#-- parse command line arguments --#
while getopts Gc:d:i:p:u:vxh? arg
do
  case "${arg}" in
  c) community_name="${OPTARG}"
     ;;

  d) virtual_device="${OPTARG}"
     check_characters "${virtual_device}" "Device"
     if [ "${?}" != "0" ];then
        exit 9
     fi
     ;;

  h) usage_long
     exit
     ;;
     
  i) network="${OPTARG}"
     if [ ! -z "${check_type}" ];then
        usage_short
        option_mutually_exclusive "-i|-p"
        exit 99
     else
        check_type="network"
     fi
     ;;

  p) device_file="${OPTARG}"
     if [ ! -z "${check_type}" ];then
        usage_short
        option_mutually_exclusive "-i|-p"
        exit 99
     else
        check_type="local"
     fi
     ;;

  u) user_name="${OPTARG}"   
     ;;

  v) display_version
     exit
     ;;

  x) DEBUG=1
     ;;

  *) usage_short
     exit 99;;
  esac 
done

if [ -z "${virtual_device}" -a -z "${check_type}" -a -z "${user_name}" ];then
   usage_short
   ${PRINT_CMD} "Missing an argument, specify one or more [ -d | -i | -p | -u ]."
   exit 99
fi

if [ ! -z "${check_type}" ];then 
   case "${check_type}" in
      "local")
               check_physical_devices "${device_file}"
               type_return=${?}
               ;;
    "network")
               cnd_output=`check_network_device "${network}" "${community_name}" "FALSE"`
               type_return=${?}
               if [ "$type_return" != "0" ];then
               		cnd_output=`check_network_device "${network}" "${community_name}" "TRUE"`
			type_return=${?}
               fi               
               ;;
   esac
fi

if [ ! -z "${virtual_device}" ];then
   if [ "${DEBUG}" = "1" ]; then
      ${PRODUCT_LINK}/bin/lsdevice -d "${virtual_device}" -x >/dev/null
   else
      ${PRODUCT_LINK}/bin/lsdevice -d "${virtual_device}" >/dev/null
   fi
   lsdev_return=${?}
   if   [ "${lsdev_return}" = "0" ];then
        lsdev_return=1
   elif [ "${lsdev_return}" = "1" ];then
        lsdev_return=0
   fi
fi

if [ ! -z "${user_name}" ];then
   user_output=`${PRODUCT_LINK}/bin/ckuser -u "${user_name}"`
   user_return="${?}"
   if [ "${user_return}" != "0" ]; then
      user_return=5
   fi
fi

#-- Process return codes
returns=`${EXPR_CMD} ${lsdev_return} + ${type_return} + ${user_return}`

if [ "${DEBUG}" = "1" ];then
   {
    ${PRINT_CMD} "TYPE_RETURN   = ${type_return}"
    ${PRINT_CMD} "USER_RETURN   = ${user_return}"
    ${PRINT_CMD} "LSDEV_RETURN  = ${lsdev_return}"
    ${PRINT_CMD} "RETURNS       = ${returns}"
   } >&2
fi
 
if [ "${check_type}" = "local" ];then
   case "${returns}" in
      0)
        pdebug "All Options valid\n"
        pdebug "Exit: 0\n"
        ${PRINT_CMD} "All options are valid."
        return_code="0"
        ;;
      1)
        #-- Virtual Device Name in use
        pdebug "Exit: 10\n"
        ${PRINT_CMD} "Virtual device ${virtual_device} already exists on the system."
        return_code="10"
        ;;
      3)
        #-- Invalid Character Device
        pdebug "Exit: 11\n"
        ${PRINT_CMD} "Physical device ${device_file} is not valid character device."
        return_code="11"
        ;;
      4)
        #-- Virtual Device Name in use and Invalid Character Device
        pdebug "Exit: 13\n"
        ${PRINT_CMD} "1. Virtual device ${virtual_device} already exists on the system."
        ${PRINT_CMD} "2. Physical device ${device_file} is not valid character device."
        return_code="13"
        ;;
      5)
        #-- Invalid User
        pdebug "Exit: 11\n"
        ${PRINT_CMD} "Username ${user_name} is not a valid system user!"
        return_code="12"
        ;;
      6) 
        #-- Virtual Device Name in use and Invalid user
        pdebug "Exit: 14\n"
        ${PRINT_CMD} "1. Virtual device ${virtual_device} already exists on the system."
        ${PRINT_CMD} "2. Username ${user_name} is not a valid system user!"
        return_code="14"
        ;;
      8)
        #-- Invalid Character Device and Invalid user
        pdebug "Exit: 15\n"
        ${PRINT_CMD} "1. Physical device ${device_file} is not valid character device."
        ${PRINT_CMD} "2. Username ${user_name} is not a valid system user!"
        return_code="15"
        ;;
      9)
        #-- Virtual Device Name in use, Invalid Character Device and Invalid user
        pdebug "Exit: 16\n"
        ${PRINT_CMD} "1. Virtual device ${virtual_device} already exists on the system."
        ${PRINT_CMD} "2. Physical device ${device_file} is not valid character device."
        ${PRINT_CMD} "3. Username ${user_name} is not a valid system user!"
        return_code="16"
        ;;
      *)
        ${PRINT_CMD} "ERROR: Unknown Return Codes."
        return_code="99"
        ;;
   esac

   exit "${return_code}" 

else

   #-- Process the return codes
   if   [ ${type_return} -lt 4 -o ${type_return} -eq 44 ];then
        returns=`${EXPR_CMD} ${returns} \- ${type_return}` 
   fi
   
   case "${returns}" in
      0)
        pdebug "All Options valid\n"
        pdebug "Exit: 0\n"
        if [ -z "${network}" ];then
           ${PRINT_CMD} "All options are valid."
        else
           ${PRINT_CMD} "${cnd_output}"
        fi
        return_code="${type_return}"
        ;;
      1)
        #-- Virtual Device Name in use
        pdebug "Exit: 10\n"
        ${PRINT_CMD} "Virtual device ${virtual_device} already exists on the system."
        return_code="10"
        ;;
      4)
        #-- Unable to contact device [ Down or Unknown ]
        pdebug "Exit: 4\n"
        ${PRINT_CMD} "Print server is either down or unknown."
        return_code="4"
        ;;
      5)
        #-- User is invalid or Device in use and Unable to contact device.
        if [ "${lsdev_return}" = "1" ];then
           pdebug "Exit: 17\n"
           ${PRINT_CMD} "1. Virtual device ${virtual_device} already exists on the system."
           ${PRINT_CMD} "2. Print server is either down or unknown."
           return_code="17"
        else
           pdebug "Exit: 12\n"
           ${PRINT_CMD} "Username ${user_name} is not a valid system user!"
           return_code="12"
        fi
        ;;
      6)
        #-- Device in use and Invalid User
        pdebug "Exit: 14\n"
        ${PRINT_CMD} "1. Virtual device ${virtual_device} already exists on the system."
        ${PRINT_CMD} "2. Username ${user_name} is not a valid system user!"
        return_code="14"
        ;;
      9)
        #-- Unable to contact device and Invalid User
        pdebug "Exit: 19\n"
        ${PRINT_CMD} "1. Print server is either down or unknown."
        ${PRINT_CMD} "2. Username ${user_name} is not a valid system user!"
        return_code="19"
        ;;
     10)
        #-- Device in use, Unable to contact device and Invalid User
        pdebug "Exit: 18\n"
        ${PRINT_CMD} "1. Virtual device ${virtual_device} already exists on the system."
        ${PRINT_CMD} "2. Print server is either down or unknown."
        ${PRINT_CMD} "3. Username ${user_name} is not a valid system user!"
        return_code="18"
        ;;
    101)
        #-- Invalid Hostname or IP address 
        pdebug "Exit: 101\n"
        ${PRINT_CMD} "${cnd_output}"
        return_code="101"
        ;;
    102)
        #-- Device in use and Invalid Hostname or IP address 
        pdebug "Exit: 20\n"
        ${PRINT_CMD} "1. Virtual device ${virtual_device} already exists on the system."
        ${PRINT_CMD} "2. Invalid IP Address or Hostname"
        return_code="20"
        ;;
    104)
        #-- Invalid link-local IPv6 address
        pdebug "Exit: 104\n"
        ${PRINT_CMD} "${cnd_output}"
        return_code="104"
        ;;
    106)
        #-- Invalid Hostname or IP address and Invalid User 
        pdebug "Exit: 22\n"
        ${PRINT_CMD} "1. Invalid IP Address or Hostname"
        ${PRINT_CMD} "2. Username ${user_name} is not a valid system user!"
        return_code="22"
        ;;
    107)
        #-- Device in use, invalid user, Invalid Hostname or IP address 
        pdebug "Exit: 21\n"
        ${PRINT_CMD} "1. Virtual device ${virtual_device} already exists on the system."
        ${PRINT_CMD} "2. Invalid IP Address or Hostname"
        ${PRINT_CMD} "3. Username ${user_name} is not a valid system user!"
        return_code="21"
        ;;
      *)
        ${PRINT_CMD} "ERROR: Unknown Return Codes."
        return_code="99"
        ;;
    esac 

    exit "${return_code}" 
fi

exit 0
