#!/bin/sh

########################################################################
#             (c) Copyright 2003 Lexmark International, Inc.           #
#                        All rights reserved                           #
########################################################################
#                                                                       
# This command is a print job submission GUI. It starts up a 
# print queue properties UI that allows the user to change queue
# settings for that specific job.
#
########################################################################

#--
#-- Java VM system defines can be passed to the Java VM by specifying
#-- the defines in the VMDEFINES environment variable. To specify
#-- multiple defines, enclose the define statements in quotes. For
#-- example:
#--
#-- % export VMDEFINES="-Duser.language=de -Duser.region=DE"
#--

#-- set debug to false by default. To turn debug "on" set to "1"
DEBUG=0

#-- Trap signal 2 and 15 and exit
trap "cleanup; exit 0" 1 2 15


PRODUCT_LINK="/usr/genprint"
SCRIPTS_DIR="${PRODUCT_LINK}/bin/.scripts"

#-- Load the shared script
. ${SCRIPTS_DIR}/shared

cleanup()
{
   if [ "${file_location}" = "stdin" ];then
      ${PRINTF_CMD} "Cleaning Up .. " >>${DEBUG_LOG} 2>&1
      ${RM_CMD} /tmp/${PRINT_GUI}_job.$$ 
      ${PRINTF_CMD} "done.\n" >>${DEBUG_LOG} 2>&1
   fi 
}


#-- Initialize
initialize
#-- initialize classpath
set_classpath

#-- Save DISPLAY
display=${DISPLAY}

program_name=`${BASENAME_CMD} $0 | ${SED_CMD} -e 's/\.sh//'`

#-- Need special Java VM flags for languages, override the
#-- VMDEFINES variable.
if [ -z "${VMDEFINES}" ];then
   if [ ! -z "${APPLICATION_LANG}" ];then
      LANGUAGE="`${PRINT_CMD} ${APPLICATION_LANG} | ${AWK_CMD}  -F'_' '{print $1}' | ${SED_CMD} -e 's/\..*$//'`"
      REGION="`${PRINT_CMD} ${APPLICATION_LANG} | ${AWK_CMD} -F'_' '{print $2}' | ${SED_CMD} -e 's/\..*$//'`"
      if [ "${REGION}" = "" ];then
         REGION="${LANGUAGE}"
      fi
      VMDEFINES="-Duser.language=${LANGUAGE} -Duser.region=${REGION}"
      LC_ALL=${APPLICATION_LANG}
      export LC_ALL
   fi
fi

#-- usage_short text
usage_short ()
{
${CAT_CMD} << EOT

USAGE:
    ${program_name} -d queue [-display display] [-x] filename
    ${program_name} -h
    ${program_name} -v

Try '${program_name} -h' for more information.

EOT
}


#-- usage_long text
usage_long ()
{
${CAT_CMD} << EOT

${program_name} 
    Launches a printer properties window that gives a user the ability to change
    printer related options before submitting a print job.

USAGE:
    ${program_name} -d queue [-display display] [-x] filename
    ${program_name} -h
    ${program_name} -v

OPTIONS:
    -d queue
       Specify the target printer queue.

    -display display
       Specify the X display on which to run.

    -x
       Run ${program_name} in debug mode.

    -h
       Display this help and exit.

    -v
       Display version information and exit.
           
EXAMPLES:
    Print the file myfile.txt on the queue myqueue:

       $ ${program_name} -d myqueue myfile.txt

    Print the file myfile.txt on the queue myqueue, but use the first X display
    on the remote host somehost.somedomain.com:

       $ ${program_name} -d myqueue -display somehost.somedomain.com:0.0

    Print the file myfile.ps on the queue myqueue, forcing the file type to be
    PostScript:

       $ ${program_name} -d myqueue myfile.ps

EOT
}


#-- parse arguments begin
while [ ! -z "${1}" ]
do
   case "$1" in

   -display) shift 1
              OPTARG="`${PRINT_CMD} "$1" | ${CUT_CMD} -c 1`"
              if [ -z "${OPTARG}" ] || [ "${OPTARG}" = "-" ]; then
                 usage_short
                 ${PRINT_CMD} "Specify a hostname with the -d option."
                 exit 99
              else
                 display="${1}"
                 process_display ${display}
              fi
              unset OPTARG
              ;;

    -q|-P|-d) flag="${1}" ; shift 1
              OPTARG="`${PRINT_CMD} "${1}" | ${CUT_CMD} -c 1`"
              if [ -z "${OPTARG}" ] || [ "${OPTARG}" = "-" ]; then
                 usage_short
                 ${PRINT_CMD} "Specify a queue_name with the [ $flag ] option."
                 exit 99
              else
                 queue_name="${1}"

              fi
              unset OPTARG
              ;;

          -V) display_version
              exit 99
              ;;

          -v) display_version
              exit 99
              ;;

          -x) DEBUG="1" ;;

   -h|--help) usage_long; 
              exit 99 
              ;;


           *) ARGS="$ARGS $1"
              LAST_ARG="${1}"
              ;;
   esac
   shift 1
done
#-- parse arguments ends

#-- CLASSPATH is set in shared
TARGET="com.lexmark.unix.drivers.jprint.printeroptions.PrinterPropertiesUI"
MESSAGE_TARGET="com.lexmark.unix.drivers.jprint.util.ShowLPDMessages"

#-- Setup log filename
if [ ${DEBUG} = "0" ];then
   DEBUG_LOG="/dev/null"
else
   DEBUG_LOG="/tmp/${program_name}_main.$$"
   ${PRINT_CMD} "Debug Mode:  Everything is being logged to ${DEBUG_LOG}"
fi

#-- Check to see if regular users have access to the spool directories.
check_spool_dir_permissions >/dev/null 2>&1
if [ "${?}" != "0" ];then
   ${JAVA_CMD} ${VMDEFINES} -cp ${CLASSPATH} ${MESSAGE_TARGET} 14 >> ${DEBUG_LOG} 2>&1
   exit 1
fi

#-- the last argument should be a file, otherwise the job
#-- should be provided via STDIN.
filename="${LAST_ARG}"

#-- Check for License Agreement
license_agreement "user" "show_gui"

if [ -z "${queue_name}" ];then 
   usage_short
   ${PRINT_CMD} "Missing option, please specify a queue name."
   exit 99
else
   check_queue_status ${queue_name} >>${DEBUG_LOG} 2>&1
   if [ "${?}" != "0" ];then
      ${PRINT_CMD} "Unknown printer queue [ ${queue_name} ]."
      exit 1
   fi
   get_queue_settings ${VIR_QUEUES_DIR}/${queue_name}/.${queue_name} >>${DEBUG_LOG} 2>&1
   if [ "${?}" != "0" ];then
      ${PRINT_CMD} "Printer queue [ ${queue_name} ] is not a ${APPLICATION_NAME} queue."
      ${PRINT_CMD} "Run lsqueue -l to see a list of available queues."
      exit 1
   fi
fi

if [ -z "${filename}" ] || [ ! -f "${filename}" ];then
   # Job should be from STDIN
   # 
   filename=/tmp/${PRINT_GUI}_job.$$
   file_location="stdin"
   ${CAT_CMD} >/tmp/${PRINT_GUI}_job.$$
else
   file_location="system"
fi 

${PRODUCT_LINK}/bin/sniff "${filename}"
sniff_rc="${?}"
if   [ "${sniff_rc}" = "0" ];then
     pdebug "\nFile Type is pcl\n" >>${DEBUG_LOG} 2>&1 
     job_type="pcl" 
elif [ "${sniff_rc}" = "1" ];then
     pdebug "\nFile Type is ps\n" >>${DEBUG_LOG} 2>&1
     job_type="ps" 
elif [ "${sniff_rc}" = "2" ];then
     pdebug "\nFile Type is iq\n" >>${DEBUG_LOG} 2>&1
     job_type="iq" 
else
     pdebug "\nFile Type is unknown, default to pcl\n" >>${DEBUG_LOG} 2>&1
     job_type="pcl" 
fi

show_mismatch="no"
queueSupport=`get_queue_emulation ${queue_name}`
if   [ "${queueSupport}" = "passthrough" ];then
	job_type="NONE"
elif [ "${queueSupport}" = "pcl" -a "${job_type}" != "pcl" ];then
	show_mismatch="yes"
	job_type="pcl" 
elif [ "${queueSupport}" = "ps" -a "${job_type}" != "ps" ];then
	show_mismatch="yes"
	job_type="ps" 
elif [ "${queueSupport}" = "iq" -a "${job_type}" != "iq" ];then
	show_mismatch="yes"
	job_type="iq" 
elif [ "${queueSupport}" = "pdf" -a "${job_type}" != "iq" ];then
	show_mismatch="yes"
	job_type="pdf" 
elif [ "${queueSupport}" = "automatic" -a "${job_type}" = "iq" ];then
	show_mismatch="yes"
	job_type="pcl" 
elif [ "${queueSupport}" = "ppds" ];then
	if [ "${job_type}" != "pcl" ];then
		show_mismatch="yes"
	fi
	job_type="ppds" 
fi

#-- Check display
check_display

if [ "${job_type}" != "NONE" ];then
   ${PRINT_CMD} "Launching Printer Options ... " >>${DEBUG_LOG} 2>&1

	if [ "${show_mismatch}" = "yes" ];then
		pdebug "${JAVA_CMD} ${VMDEFINES} -cp ${CLASSPATH} ${MESSAGE_TARGET} 7\n" >>${DEBUG_LOG} 2>&1
		${JAVA_CMD} ${VMDEFINES} -cp ${CLASSPATH} ${MESSAGE_TARGET} 7 >>${DEBUG_LOG} 2>&1
		if [ "${?}" != "0" ];then
   		${PRINTF_CMD} "\nPrinting cancelled.\n" 
   		if [ "${file_location}" != "system" ];then 
      		cleanup
   		fi
   		exit 1
		fi
	fi

   pdebug "${JAVA_CMD} ${VMDEFINES} -cp ${CLASSPATH} ${TARGET} ${queue_name} ${job_type}\n" >>${DEBUG_LOG} 2>&1
   PRINT_ARGS=`${JAVA_CMD} ${VMDEFINES} -cp ${CLASSPATH} ${TARGET} ${queue_name} ${job_type} 2>/dev/null | ${TAIL_CMD} -1`

   pdebug "\nCaught args = ${PRINT_ARGS} \n" >>${DEBUG_LOG} 2>&1
fi

print_command="`build_print_command ${queue_name} ${PRINT_ARGS} 2>>${DEBUG_LOG}`"
if [ "${?}" != "0" ];then
   ${PRINTF_CMD} "\nPrinting cancelled.\n" 
   if [ "${file_location}" != "system" ];then 
      cleanup
   fi
   exit 1
fi

if [ "${file_location}" = "system" ];then
   #-- Replaces spaces so that it can print filenames with spaces
   args=`${PRINT_CMD} ${ARGS} | ${SED_CMD} -e 's/ /\\\ /g'`
   pdebug "${print_command} ${args}\n" >>${DEBUG_LOG} 2>&1 
   eval ${print_command} ${args}

else

   pdebug "${CAT_CMD} ${filename} | ${print_command} ${ARGS}\n" >>${DEBUG_LOG} 2>&1
   filename=`${PRINT_CMD} ${filename} | ${SED_CMD} -e 's/ /\\\ /g'`
   ${CAT_CMD} ${filename} | eval ${print_command} ${ARGS}
   ${RM_CMD} /tmp/${PRINT_GUI}_job.$$

fi

exit 0

