#!/bin/sh

########################################################################
#             (c) Copyright 2004 Lexmark International, Inc.           #
#                        All rights reserved                           #
########################################################################
#                                                                       
# This command is used to create a virtual device.
#
########################################################################
#
# Exit codes:
#
#  0  --- Success - Creating virtual device was successful
#  1  --- Error   - Creating virtual device failed 
# 99 --- Error   - Command line options incorrect
#
########################################################################

#-- set debug to false by default. To turn debug "on" set to "1"
DEBUG=0

#-- set initial return_code
return_code=99

#-- initialize variables to 0
type_return=0
bpp_device=2


PRODUCT_LINK="/usr/genprint"
SCRIPTS_DIR="${PRODUCT_LINK}/bin/.scripts"

#-- load the shared script
. ${SCRIPTS_DIR}/shared

#-- initialize
initialize

program_name="mkdevice"
SHOW_PORTS="parallel, usb or serial"

#-- usage_short text
usage_short ()
{
${CAT_CMD} << EOT

USAGE:
    ${program_name} -d name -f device -t type [-x]
    ${program_name} -d name -i host [-c name] [-p port] [-t type] [-x]
    ${program_name} -h
    ${program_name} -v

Try '${program_name} -h' for more information

EOT
}

#-- usage_long text
usage_long ()
{
${CAT_CMD} << EOT

${program_name}
    Creates a virtual device. A virtual device contains the information a queue
    needs to deliver data to the printer.  A network virtual device, for
    example, tells the queue that the printer is connected through a network
    connection and where on the network the printer is located.

USAGE:
    ${program_name} -d name -f device -t type [-x]
    ${program_name} -d name -i host [-c name] [-p port] [-t type] [-x]
    ${program_name} -h
    ${program_name} -v

OPTIONS:
    -d name
       Specify the name of the new virtual device.

    -f device
       Specify the absolute device filename for a local device.

    -t type
       Specify the connection type.
       Valid local device types are ${SHOW_PORTS}.
       Valid external network device types are 1-port, 2-port, 3-port or 4-port.

    -p port
       Specify the external print server's port number for networked devices.
       If the adapter specified by the -i option is down or non-existent,
       passing a '-p 1' will allow successful creation of the device.

    -i host
       Specify the hostname or IPv4/IPv6 address. 
       If you are specifying an IPv6 link-local address, you need to specify the
       interface name using the following format: ipv6_address%interface_name
       Example: fe80::204:ff:fe13:cce5%eth0
        
    -c name
       Specify the SNMP community name.  The default value is 'public'.

    -x
       Run ${program_name} in debug mode.

    -h
       Display this help and exit.

    -v
       Display version information and exit.

EXAMPLES:
    Create a virtual device named 'mydevice' pointing to the local
    parallel device '/dev/lp0':

      $ ${program_name} -d mydevice -f /dev/lp0 -t parallel

    Create a virtual device named 'mydevice' pointing to the 2nd port of the
    external print server host 'server.domain.com':

      $ ${program_name} -d mydevice -i server.domain.com -p 2

    Create a virtual device named 'mydevice' pointing to the print server host
    'server.domain.com' using the SNMP community name 'test':

      $ ${program_name} -d mydevice -i server.domain.com -c test

    Create a virtual device named 'mydevice' pointing to a non-existent print
    server host named 'nonexistent.domain.com':

      $ ${program_name} -d mydevice -i nonexistent.domain.com -p 1

EOT
}

write_local()
{
   dname="${1}"
   ctype="${2}"
   dfile="${3}"
   odevice="${4}" 
   bppdevice="${5}"

   #-- BEGIN WRITING DEVICE FILE
   {
   ${PRINT_CMD} "device_name=${dname}"
   ${PRINT_CMD} "device_desc=Virtual Device"
   ${PRINT_CMD} "transport_type=${ctype}"

   if [ "${odevice}" = "true" ];then
      ${PRINT_CMD} "${ctype}_device=other"
      ${PRINT_CMD} "other_${ctype}_device=${dfile}"
   else
      ${PRINT_CMD} "${ctype}_device=${dfile}"
   fi
 
   if [ "${ctype}" = "serial" ];then
      ${PRINT_CMD} "parity=none"
      ${PRINT_CMD} "serial_protocol=xon_xoff"
      ${PRINT_CMD} "data_bits=8"
      ${PRINT_CMD} "baud_rate=9600"
   fi
   } > "${VIR_DEVICES_DIR}/${dname}"

   ${CHMOD_CMD} ${SPOOL_FILE_PERMS} "${VIR_DEVICES_DIR}/${dname}"
   ${CHOWN_CMD} lp:lp "${VIR_DEVICES_DIR}/${dname}"
   #-- END WRITING DEVICE FILE
   return ${?}
}

write_network()
{
   dname="${1}"
   host="${2}"
   cname="${3}" 
   external="${4}"
   port="${5}"
   no_of_ports="${6}"
  
   if [ -z "${no_of_ports}" ]; then
      if [ "${port}" = "1" ];then
         no_of_ports=1
      elif [ "${port}" = "9" ];then
         no_of_ports=3
      elif [ "${port}" = "3" -o "${port}" = "4" ];then
         no_of_ports=4
      elif [ "${connection}" = "3-port" ];then
         no_of_ports=3
      elif [ "${connection}" = "4-port" ];then
         no_of_ports=4
      else
         no_of_ports=2
      fi
   fi

   #-- BEGIN WRITING DEVICE FILE
   {
   ${PRINT_CMD} "device_name=${dname}"
   ${PRINT_CMD} "device_desc=Virtual Device"
   ${PRINT_CMD} "transport_type=network"
   ${PRINT_CMD} "network_host=${host}"
   ${PRINT_CMD} "read_net_host_comm=${cname}"
   ${PRINT_CMD} "write_net_host_comm=${cname}"

   if [ "${external}" = "true" ];then
      ${PRINT_CMD} "raptor_ports=${port}"
      ${PRINT_CMD} "no_of_ports=${no_of_ports}"
   fi
   
   ${PRINT_CMD} "administrator=root"
   ${PRINT_CMD} "logfile=/tmp/logfile"
   ${PRINT_CMD} "ir_user=mail"
   ${PRINT_CMD} "ir_admin=mail"
   ${PRINT_CMD} "pr_user=none"
   ${PRINT_CMD} "pr_admin=none"
   } > "${VIR_DEVICES_DIR}/${dname}"

   ${CHMOD_CMD} ${SPOOL_FILE_PERMS} "${VIR_DEVICES_DIR}/${dname}"
   ${CHOWN_CMD} lp:lp "${VIR_DEVICES_DIR}/${dname}"
   #-- END WRITING DEVICE FILE
   return ${?}
}


#-- Check for Admin user
is_privileged_user
if [ "${?}" != "0" ];then
   ${PRINT_CMD}
   ${PRINT_CMD} "You must be an administrator to run this program."
   ${PRINT_CMD}
   exit 99
fi


#-- MAIN --#
#-- parse command line arguments --#
while getopts c:d:f:i:p:t:vxh? arg
do
  case "${arg}" in

  c) community_name="`${PRINT_CMD} ${OPTARG} | ${AWK_CMD} '{ print $1 }'`"
     ;;

  d) virtual_device="${OPTARG}"
     ;;

  h) usage_long
     exit 99
     ;;
     
  i) network="${OPTARG}"
     if   [ ! -z "${check_type}" ];then
          usage_short
          option_mutually_exclusive "-i|-f"
          exit 99
     else
          check_type="network"
     fi
     ;;

  f) device_file="${OPTARG}"
     if    [ ! -z "${check_type}" ];then
           usage_short
           option_mutually_exclusive "-i|-f"
           exit 99
     elif  [ ! -z "${port}" ];then
           usage_short
           ${PRINT_CMD} "The [-p] option should not be used with the [-f] option."
           exit 99
     else
        check_type="local"
     fi
     ;;

  p) port="${OPTARG}"
     if [ ! -z "${device_file}" ];then
        usage_short
        exit 99
     fi 
     ;; 

  t) connection="${OPTARG}"
     ;;

  v) display_version
     exit 99
     ;;

  x) DEBUG=1
     ;;

  *) usage_short
     exit 99
     ;;
  esac 
done

#-- Require virtual device name
if [ -z "${virtual_device}" ];then
   usage_short
   ${PRINT_CMD} "Missing [-d] option, specify a device name."
   exit 99
fi 

#-- Require check device type
if [ -z "${check_type}" ];then
   usage_short
   ${PRINT_CMD} "Missing [-i or -f] option."
   exit 99
fi

#-- Check for valid connection type
if [  ! -z "${connection}" -a ! -z "${network}" ];then
	if [  "${connection}" != "1-port" -a  "${connection}" != "2-port" -a "${connection}" != "3-port" -a  "${connection}" != "4-port" ];then
		usage_short
		${PRINT_CMD} "Invalid type [ ${connection} ]. Valid types are 1-port, 2-port, 3-port or 4-port."
		exit 99
	fi
elif [  ! -z "${connection}" -a ! -z "${device_file}" ];then
	if [ "${connection}" != "parallel" -a  "${connection}" != "serial" -a  "${connection}" != "usb" ];then
		usage_short
		${PRINT_CMD} "Invalid type [ ${connection} ]. Valid types are ${SHOW_PORTS}."
		exit 99
	fi
fi

if [ ! -d "${VIR_DEVICES_DIR}" ];then
	${PRINT_CMD} "ERROR:  The virtual device directory is missing."
	${PRINT_CMD}
	${PRINT_CMD} "Run the [ setup.${PRODUCT_NAME} ] script to recreate the appropriate"
	${PRINT_CMD} "application directories."
	${PRINT_CMD}
	exit 2 
fi

#-- Check device settings.
if [ "${check_type}" = "local" ];then
   if [ -z "${connection}" ];then 
      usage_short
      ${PRINT_CMD} "Missing [-t] option, specify a connection type."
      exit 99
   else
      if [ "${DEBUG}" = "0" ]; then
	 type_output=`${PRODUCT_LINK}/bin/ckdevice -d "${virtual_device}" -p "${device_file}"`
	 type_return=${?}
      else
         ${PRINT_CMD} "Running -- ${PRODUCT_LINK}/bin/ckdevice -d \"${virtual_device}\" -p \"${device_file}\" -x"
	 type_output=`${PRODUCT_LINK}/bin/ckdevice -d "${virtual_device}" -p "${device_file}" -x`
	 type_return=${?}
      fi
   fi
elif [ "${check_type}" = "network" ];then
   #-- Set community name option if variable is not null
   if [ ! -z "${community_name}" ];then
      cname="-c ${community_name}"
   else
      cname=""
   fi
   if [ "${DEBUG}" = "0" ]; then
      type_output=`${PRODUCT_LINK}/bin/ckdevice -d "${virtual_device}" -i "${network}" "${cname}"`
      type_return=${?}
   else
      ${PRINT_CMD} "Running -- ${PRODUCT_LINK}/bin/ckdevice -d \"${virtual_device}\" -i \"${network}\" ${cname} -x"
      type_output=`${PRODUCT_LINK}/bin/ckdevice -d "${virtual_device}" -i "${network}" "${cname}" -x`
      type_return=${?}
      ${PRINTF_CMD} "${type_output}"
   fi

	if [ ! -z "${port}" ];then
		if [ "${type_return}" = "44" ];then
			${PRINT_CMD} ${port} | ${EGREP_CMD} -e "^[1234]$" >/dev/null 2>&1
			if [ "${?}" != "0" ];then
	 			${PRINT_CMD} "Invalid port number [ ${port} ]. Valid ports are 1, 2, 3, or 4."
				exit 99
			fi
		else
			${PRINT_CMD} ${port} | ${EGREP_CMD} -e "^[12349]$" >/dev/null 2>&1
			if [ "${?}" != "0" ];then
	 			${PRINT_CMD} "Invalid port number [ ${port} ]. Valid ports are 1, 2, 3, 4 or 9."
				exit 99
			fi
		fi
	fi
fi

#-- Process type_return return code
case "${type_return}" in 

   0)
      external="false"
      ;;
   1)
      external="false"
      ;;
   2) 
      if [ -z "${port}" ]; then 
	 ${PRINT_CMD} "Found external print server, you must specify a port number."
	 ${PRINT_CMD} "Valid port numbers are 1 or 2."
	 exit 1; 
      fi
      external="true"
      no_of_ports=2
      ;;
   3) 
      if [ -z "${port}" ]; then 
	 ${PRINT_CMD} "Found external print server, you must specify a port number."
	 ${PRINT_CMD} "Valid port numbers are 1, 2, or 9."
	 exit 1; 
      fi
      external="true"
      no_of_ports=3
      ;;
	44)
      if [ -z "${port}" ]; then 
	 		${PRINT_CMD} "Found external print server, you must specify a port number."
	 		${PRINT_CMD} "Valid port numbers are 1, 2, 3, 4."
	 		exit 1; 
      fi
      external="true"
      no_of_ports=4
      ;;
   4) 
      if [ -z "${port}" ]; then 
	 ${PRINT_CMD} "Print Server unknown or down, you must specify a port number."
	 exit 1; 
      fi
      external="true"
      ;;
 99)
      ${PRINTF_CMD} "${type_output}"
      exit 99 
      ;;
   *)
      ${PRINTF_CMD} "ERROR:\n${type_output}\n"
      exit "${type_return}" 
      ;;
esac

if [ "${check_type}" = "local" ];then
   #-- Check to see if this is a known port 
   ${PRODUCT_LINK}/bin/lsdevice -p | ${GREP_CMD} "${connection}#${device_file}" >/dev/null 2>&1
   if [ "${?}" = "0" ];then
      other_device="false"
   else
      other_device="true"
   fi

   write_local "${virtual_device}" "${connection}" "${device_file}" "${other_device}" "${bpp_device}"
   write_return_code="${?}"
else
   if [ -z "${community_name}" ];then
      community_name="public"
   fi
   write_network "${virtual_device}" "${network}" "${community_name}" "${external}" "${port}" "${no_of_ports}"
   write_return_code="${?}"
fi

if [ "${write_return_code}" != "0" ];then
   ${PRINT_CMD} "Creating virtual device [ ${virtual_device} ] failed!"
   exit 1
else
   ${PRINT_CMD} "Creating virtual device [ ${virtual_device} ] was successful."
   exit 0
fi

exit 0
