#!/bin/sh

########################################################################
#             (c) Copyright 2003 Lexmark International, Inc.           #
#                        All rights reserved                           #
########################################################################
#
#   Script to check the status cups and lpd.
#
#   EXIT 0  --  LPD   RUNNING 
#   EXIT 1  --  CUPSD RUNNING 
#   EXIT 2  --  CUPSD FOUND BUT NOT RUNNING
#   EXIT 3  --  LPD   FOUND BUT NOT RUNNING
#   EXIT 4  --  LPD and CUPS NOT FOUND
#   EXIT 5  --  LPD and CUPS BOTH FOUND
#
########################################################################

#-- initialize variables
START="0"
CUPS_FOUND="0"
LPD_FOUND="0"


PRODUCT_LINK="/usr/genprint"
SCRIPTS_DIR="${PRODUCT_LINK}/bin/.scripts"

#-- load the shared script
. ${SCRIPTS_DIR}/shared

#-- Only initialize admin if we need to start the daemon.
initialize user

#-- usage_short text
usage_short ()
{
program_name=`${BASENAME_CMD} $0`

${CAT_CMD} << EOT

USAGE:
    ${program_name} [-s]
    ${program_name} -h
    ${program_name} -v

OPTIONS:
    -s
       Start the Print Daemon if found and not running.

    -h
       Display this help and exit.

    -v
       Display version information and exit.

EOT
}

check_for_root()
{
   is_privileged_user
   if [ $? -ne 0 ];then 
      ${PRINT_CMD} "You must be ROOT to run `basename $0` with the -s flag." 
      exit 99
   fi
}

is_lpd_running () 
{
	PID=`${PS_CMD} auxw | ${GREP_CMD} lpd | ${GREP_CMD} -v grep | ${AWK_CMD} '{print $2}'`
	if [ -z "${PID}" ]; then return 1; fi
}

is_cupsd_running () 
{
	PID=`${PS_CMD} auxw | ${GREP_CMD} cupsd | ${GREP_CMD} -v grep | ${AWK_CMD} '{print $2}'`
	if [ -z "${PID}" ]; then return 1; fi
}

is_redhat_cups()
{
       if [ -s /etc/alternatives/print-lpc ];then
          ${LS_CMD} -l /etc/alternatives/print-lpc | ${GREP_CMD} -i cups >/dev/null 2>&1   
          if [ $? -eq 0 ];then
             return 0
          else
             return 1
          fi
       else
             return 2
       fi
}

manage_cupsd()
{
	${PRINT_CMD} "CUPSD is not running, but it can be started."
	if [ "${START}" = "1" ];then
           start_cups
        else
           exit 2
        fi
}

start_cups()
{
      check_for_root

		#-- Set Path for SuSE
		PATH=/sbin:${PATH}
		export PATH
      ${CUPSD_START_CMD} >/dev/null 2>&1
      is_cupsd_running
      if [ "${?}" = "0" ]; then
         ${PRINT_CMD} "Success: CUPSD started !"
         exit 0
      else
         ${PRINT_CMD} "Error: CUPSD failed to start. [ ${CUPSD_START_CMD} ] !"
         exit 1
      fi
}

manage_lpd()
{
	${PRINT_CMD} "LPD is not running, but it can be started."
	if [ "${START}" = "1" ];then
           start_lpd
        else
           exit 3
        fi
}

start_lpd()
{
      check_for_root

      ${LPD_CMD} >/dev/null 2>&1
      is_lpd_running
      if [ "${?}" = "0" ]; then
         ${PRINT_CMD} "Success: LPD started !"
         exit 0
      else
         ${PRINT_CMD} "Error: LPD failed to start. [ ${LPD_CMD} ] !"
         exit 2
      fi
}

manage_both_daemons()
{	
	if [ "${START}" = "0" ];then
           ${PRINT_CMD} "CUPSD and LPD was found."
           exit 5 
        fi

        PROP_FILE="${ROOT_SPOOL_DIR}/.${PRODUCT_NAME}/props/app.properties"
        if [ -f "${PROP_FILE}" ];then
             ${CAT_CMD} ${PROP_FILE} | ${GREP_CMD} "app.prt_subsystem=cups" >/dev/null 2>&1
             if [ ${?} -eq 0 ];then
                manage_cupsd
             else
                manage_lpd
             fi
        else
             ${PRINT_CMD} "CUPSD and LPD was found."
             exit 5 
        fi
}

while getopts hsv OPTS
do
	case "$OPTS" in
        	h) 	usage_short
           	   	exit 99
           	   	;; 

		s)	START=1;;

      		v) 	display_version
           		exit 99
           		;; 
	esac
done


#-- Lets take a look for CUPSD Print Subsytem. 
if [ -x "${CUPSD_CMD}" ]; then
     is_cupsd_running
     rc=${?}
     is_redhat_cups
     rc2=${?}
     if [ ${rc} -eq 0 ] && [ ${rc2} -eq 0 -o ${rc2} -eq 2 ];then
	${PRINT_CMD} "CUPSD seems to be running."
	exit 1
     else
        CUPS_FOUND="1" 
     fi
fi

#-- Lets take a look for LPD Print Subsytem. 
if [ -x "${LPD_CMD}" ]; then
     is_lpd_running
     rc=${?}
     is_redhat_cups
     rc2=${?}
     if [ ${rc} -eq 0 ] && [ ${rc2} -eq 1 -o ${rc2} -eq 2 ];then
	${PRINT_CMD} "LPD seems to be running."
	exit 0
     else
        LPD_FOUND="1" 
     fi
fi

#-- Lets evaluate what we found.
if   [ "${CUPS_FOUND}" = "0" -a "${LPD_FOUND}" = "0" ];then
       #-- We could not find a valid print subsystem 
       ${PRINT_CMD} "CUPSD and LPD are missing"
       exit 4 
elif [ "${CUPS_FOUND}" = "1" -a "${LPD_FOUND}" = "1" ];then
       #-- We found both print subsystem 
       #-- Lets evaluate some more
       if [ -s /etc/alternatives/print-lpc ];then
          ${LS_CMD} -l /etc/alternatives/print-lpc | ${GREP_CMD} -i cups >/dev/null 2>&1   
          if [ $? -eq 0 ];then
             manage_cupsd
          else
             manage_lpd
          fi
       else
             manage_both_daemons
       fi
elif [ "${CUPS_FOUND}" = "1" ];then
       manage_cupsd
elif [ "${LPD_FOUND}" = "1" ];then
       manage_lpd
fi

