#!/bin/sh

########################################################################
#             (c) Copyright 2002 Lexmark International, Inc.           #
#                        All rights reserved                           #
########################################################################
#                                                                       
# This command is used to remove printer queues.
#
########################################################################
#
# Exit Codes:
#
#  0  --- Success  - Printer queue removed successfully
#  1  --- Error    - Printer Queue does not exist
#  2  --- Error    - Printer Queue is not idle
#  9  --- Error    - Status command failed
# 
# 
# 99  --- Error    - Usage Error
#
########################################################################

#-- set debug to false by default. To turn debug "on" set to "1"
DEBUG=0

#-- initialize variables 
completely_remove="true"


PRODUCT_LINK="/usr/genprint"
SCRIPTS_DIR="${PRODUCT_LINK}/bin/.scripts"

#-- load the shared script
. ${SCRIPTS_DIR}/shared

#-- initialize
initialize admin

program_name="rmqueue"

#-- Trap signal 2 and 15
trap 2 15

#-- usage_short text
usage_short ()
{
${CAT_CMD} << EOT

USAGE:
    ${program_name} -q queue [-x] ${SARG}
    ${program_name} [-h]
    ${program_name} [-v]

Try '${program_name} -h' for more information

EOT
}

#-- usage_long text
usage_long ()
{
${CAT_CMD} << EOT

${program_name}
    Removes printer queues.

USAGE:
    ${program_name} -q queue [-x] ${SARG}
    ${program_name} [-h]
    ${program_name} [-v]

OPTIONS:
    -q queue
       Specify the printer queue name.

    -x
       Run ${program_name} in debug mode.

    -h
       Display this help and exit.
    -v
       Display version information and exit.

EXAMPLES:
    Remove the queue myqueue:

       $ ${program_name} -q myqueue

EOT
}

#-- Check for Admin user
is_privileged_user
if [ "${?}" != "0" ];then
   ${PRINT_CMD}
   ${PRINT_CMD} "You must be an administrator to run this program."
   ${PRINT_CMD}
   exit 99
fi

#-- Remove printer classes that are associated with this queue.
remove_classes()
{
    CLASS_LIST=`${PRODUCT_LINK}/bin/lsqueue -G -a | ${EGREP_CMD} ":${queue_name}:|:${queue_name}$"`
    for class in ${CLASS_LIST}
    do
        classname=`${PRINT_CMD} ${class} | ${AWK_CMD} -F":" '{ print $1 }'`
        cq_out=`${PRODUCT_LINK}/bin/chqueue -G -q ${queue_name} -c -r ${classname}`
		  pdebug "${cq_out}\n"
    done
}


#-- MAIN --#
#-- parse command line arguments --#
while getopts Gq:xsvh? arg
do
  case "${arg}" in
  G) completely_remove="false"
	  ASKSHUTDOWN="no"
     ;;

  h) usage_long
     exit 99
     ;;

  q) queue_name="${OPTARG}"
     ;; 

  s) ASKSHUTDOWN="no"
     ;;

  v) display_version
     exit 99
     ;;

  x) DEBUG=1
     ;;

  *) usage_short
     exit 99
     ;;
  esac 
done

if   [ -z "${queue_name}" ];then
     usage_short
     ${PRINT_CMD} "Missing [-q] option, please specify a queue name."
     exit 99 
fi


pdebug "\nStep 1: Verify the existance of the virtual queue file.\n"
pdebug "---------------------------------------------------------\n"

if [ ! -f "${VIR_QUEUES_DIR}/${queue_name}/.${queue_name}" ];then
   ${PRINT_CMD} "Error: Printer queue [ ${queue_name} ] does not exist or the queue"
   ${PRINT_CMD} "is not defined by ${APPLICATION_NAME}." 
   exit 1
fi

pdebug "Virtual queue found.\n"
pdebug "FILE=${VIR_QUEUES_DIR}/${queue_name}/.${queue_name}\n"
pdebug "..Continue..\n\n"


pdebug "\nStep 2: Verify the printer queue exists on the system.\n"
pdebug "--------------------------------------------------------\n"

cqs_out=`check_queue_status ${queue_name} 2>&1`
if [ "${?}" != "0" ];then
   if [ "${DEBUG}" = "1" ];then
      ${PRINTF_CMD} "Check Queue Status Output\n" >&2
      ${PRINTF_CMD} "-------------------------\n" >&2
      ${PRINTF_CMD} "${cqs_out}\n" >&2
      ${PRINTF_CMD} "-------------------------\n" >&2
   fi
   ${PRINT_CMD} "Error: Printer queue [ ${queue_name} ] does not exist."
   exit 1
fi
if [ "${DEBUG}" = "1" ];then
   ${PRINTF_CMD} "Check Queue Status Output\n" >&2
   ${PRINTF_CMD} "-------------------------\n" >&2
   ${PRINTF_CMD} "${cqs_out}\n" >&2
   ${PRINTF_CMD} "-------------------------\n" >&2
fi

pdebug "Printer queue [ ${queue_name} ] found.\n"
pdebug "..Continue..\n\n"

pdebug "\nStep 3: Verify the printer queue is idle.\n"
pdebug "-------------------------------------------\n"

queue_empty="false"
cq_out=`${PRODUCT_LINK}/bin/chqueue -G -q ${queue_name} -m reject`
pdebug "${cq_out}\n"

lp_status_output=`${LPSTAT_CMD} ${queue_name}`
if [ "${?}" != "0" ];then
   ${PRINT_CMD} "Error: ${LPSTAT_CMD} command failed."
   cq_out=`${PRODUCT_LINK}/bin/chqueue -G -q ${queue_name} -m accept`
	pdebug "${cq_out}\n"
   exit 9 
fi

if [ "${lp_status_output}" = "" ];then
   pdebug "The [ ${LPSTAT_CMD} ${queue_name} ] command returned nothing.\n"
   pdebug "The printer queue is empty.\n"
   pdebug "..Continue..\n\n"
   queue_empty="true"
else
   if [ ${DEBUG} = "1" ];then 
      ${PRINTF_CMD} "Status command returned output, lets check it\n" >&2
      ${PRINTF_CMD} "Status command output\n" >&2 
      ${PRINTF_CMD} "-------------------------\n" >&2
      ${PRINTF_CMD} "${lp_status_output}\n" >&2
      ${PRINTF_CMD} "-------------------------\n" >&2
   fi    
   ${PRINTF_CMD} "${lp_status_output}\n" | ${SED_CMD} 1q | ${GREP_CMD} "no entries" >/dev/null 2>&1
   if [ "${?}" = "0" ];then
      pdebug "Found no entries, printer queue is idle.\n" 
      pdebug "..Continue..\n\n"
      queue_empty="true"
   fi
  
   if [ "${queue_empty}" = "false" ];then
      cq_out=`${PRODUCT_LINK}/bin/chqueue -G -q ${queue_name} -m accept`
		pdebug "${cq_out}\n"

      ${PRINT_CMD} "There are jobs currently queued for printer queue ${queue_name}."
      ${PRINT_CMD} "Printer queues must be idle before they can be removed." 
      exit 2
   fi
fi


pdebug "\nStep 4: Remove the printer queue.\n"
pdebug "-----------------------------------\n"


#-- Classes must be removed before deleting the printer queue.
remove_classes

${LPADMIN_CMD} -x${queue_name}
lpadmin_rc="${?}"


if [ "${lpadmin_rc}" != "0" ];then
   ${PRINT_CMD} "Removing printer queue ${queue_name} failed."
   exit 1
fi

if [ ${completely_remove} = "true" ];then
   pdebug "Printer queue was removed ... [${LPADMIN_CMD} -x${queue_name} ].\n" 
   pdebug "Removing spool directory\n"
   cd ${VIR_QUEUES_DIR}
   ${RM_CMD} -rf ${queue_name} 
   if [ "${?}" != "0" ];then
      ${PRINT_CMD} "Printer queue ${queue_name} was removed successfully."
      ${PRINT_CMD} "However, the spool directory was not removed.  You will"
      ${PRINT_CMD} "need to remove it manually."
   else
      ${PRINT_CMD} "Printer queue ${queue_name} was removed successfully."
   fi
else
   ${PRINT_CMD} "Printer queue ${queue_name} was removed successfully."
fi

exit 0

