#!/bin/sh

########################################################################
#             (c) Copyright 2002 Lexmark International, Inc.           #
#                        All rights reserved                           #
########################################################################
#
# This command is used to remove printer queues.
#
########################################################################
#
# Exit Codes:
#
#  0  ---  Success -  Printer queue removed successfully
#  1  ---  Error   -  Printer Queue does not exist
#  2  ---  Error   -  Printer Queue is not idle
#  9  ---  Error   -  Status command failed
#
# 99  ---  Error   -  Usage Error
#
########################################################################

#-- set debug to false by default. To turn debug "on" set to "1"
DEBUG=0

#-- initialize variables
completely_remove="true"


PRODUCT_LINK="/usr/genprint"
SCRIPTS_DIR="${PRODUCT_LINK}/bin/.scripts"

#-- load the shared script
. ${SCRIPTS_DIR}/shared

#-- initialize
initialize admin

program_name="rmqueue"

#-- Trap signal 2 and 15
trap 2 15

#-- usage_short text
usage_short ()
{
${CAT_CMD} << EOT

USAGE:
    ${program_name} -q queue [-x]
    ${program_name} [-h]
    ${program_name} [-v]

Try '${program_name} -h' for more information

EOT
}

#-- usage_long text
usage_long ()
{
${CAT_CMD} << EOT

${program_name}
    Removes printer queues.

USAGE:
    ${program_name} -q queue [-x]
    ${program_name} [-h]
    ${program_name} [-v]

OPTIONS:
    -q queue
       Specify the printer queue name.

    -x
       Run ${program_name} in debug mode.

    -h
       Display this help and exit.

    -v
       Display version information and exit.

EXAMPLES:
    Remove the queue myqueue:

       $ ${program_name} -q myqueue 

EOT
}


remove_printcap_entry()
{

   printcap_filename=${1}

   start_line=""
   end_line=""
   num_of_lines=""
   spool_dir_line=""
   printcap_entry="valid"

   #-- Locate printcap line number of the start of queue entry
   start_line=`${GREP_CMD} -n "# >>>>>>>>>>>> Start of entry for ${queue_name} <<<<<<<<<<<<<" ${printcap_filename} | ${AWK_CMD} -F":" '{print $1}' | ${SED_CMD} 1q`

   #-- Locate printcap line number of the start of queue entry
   end_line=`${GREP_CMD} -n "# >>>>>>>>>>>> End of entry for ${queue_name} <<<<<<<<<<<<<" ${printcap_filename} | ${AWK_CMD} -F":" '{print $1}' | ${SED_CMD} 1q`

   #-- Count the number of lines between Start and End
   num_of_lines=`${PRINT_CMD} ${end_line} \- ${start_line} | ${BC_CMD}`

   #-- Locate printcap line number of the spool directory line
   spool_dir_line=`${GREP_CMD} -n ":sd=${VIR_QUEUES_DIR}/${queue_name}:" ${printcap_filename} | ${AWK_CMD} -F":" '{print $1}' | ${SED_CMD} 1q`

   if [ ${DEBUG} = "1" ]; then
      {
      ${PRINT_CMD}
      ${PRINT_CMD} "    Start Line = [${start_line}]"
      ${PRINT_CMD} "Spool Dir Line = [${spool_dir_line}]"
      ${PRINT_CMD} "      End Line = [${end_line}]"
      ${PRINT_CMD} "    Line Count = [${num_of_lines}]"
      ${PRINT_CMD}
      } >&2
   fi

   # Verify all variable have a value
   if [ -z "${start_line}" -o -z "${end_line}" -o -z "${num_of_lines}" -o -z "${spool_dir_line}" ]; then
      pdebug "A test option is null\n"
      printcap_entry="invalid"
   else
      if [ ${num_of_lines} -le 0 ]; then
         pdebug "End line is before start line.\n"
         printcap_entry="invalid"
      fi 
      if [ ${spool_dir_line} -lt ${start_line} ];then
         pdebug "Spool line is before the start line.\n"
         printcap_entry="invalid"
      fi 
      if [ ${spool_dir_line} -gt ${end_line} ]; then
         pdebug "Spool line is after the end line.\n"
         printcap_entry="invalid"
      fi 
   fi

   if [ "${printcap_entry}" = "invalid" ];then
      #-- 
      #-- Could not find a legal printcap entry
      #-- 
      ${PRINTF_CMD} "\nError - Check the ${printcap_filename} entry for inconsistencies\n" >&2
      ${PRINTF_CMD} "The entry in ${printcap_filename} was not located. No changes were made.\n" 2>&1
      ${PRINTF_CMD} "Removing printer queue ${queue_name} failed.\n"
      exit 1
   fi

   #-- We add #------ line before and after queue entry.  Check to see if we can find them.
   #-- If we find them, well remove them otherwise just continue.
   rsline=`${EXPR_CMD} ${start_line} \- 1` 
   ${SED_CMD} -n ${rsline}p ${printcap_filename} | ${EGREP_CMD} "^#--*$" >/dev/null 2>&1
   if [ "$?" = "0" ];then
      start_line=$rsline
   fi
   rsline=`${EXPR_CMD} ${end_line} + 1` 
   ${SED_CMD} -n ${rsline}p ${printcap_filename} | ${EGREP_CMD} "^#--*$" >/dev/null 2>&1
   if [ "$?" = "0" ];then
      end_line=$rsline
   fi

   pdebug "Printcap Entry is $printcap_entry\n"

   pdebug "Removing printcap entry.\n"

${ED_CMD} - ${printcap_filename} << doit
$start_line,$end_line d
w
q
doit

   if [ "${?}" != "0" ];then
      ${PRINT_CMD} "Removing printer queue ${queue_name} failed."
      exit 1
   fi
}


#-- check for lpd existance
if [ ! -x ${LPD_CMD} ];then
   ${PRINT_CMD} "This script uses a set of system commands which may be"
   ${PRINT_CMD} "in a different location on your system. The section on"
   ${PRINT_CMD} "\"System Commands\" specifies the location of these commands"
   ${PRINT_CMD}
   ${PRINT_CMD} "The following system command was not found:"
   ${PRINT_CMD}
   ${PRINT_CMD} "   ${LPD_CMD}"
   ${PRINT_CMD}
   exit 99
fi

#-- Check for Admin user
is_privileged_user
if [ "${?}" != "0" ];then
   ${PRINT_CMD}
   ${PRINT_CMD} "You must be an administrator to run this program."
   ${PRINT_CMD}
   exit 99
fi


#-- MAIN --#
#-- parse command line arguments --#
while getopts Gq:xsvh? arg
do
  case "${arg}" in
  G) completely_remove="false"
     ;;

  h) usage_long
     exit 99
     ;;

  q) queue_name="${OPTARG}"
     ;;

  v) display_version
     exit 99
     ;;

  x) DEBUG=1
     ;;

  s) IGNORE=""
     ;;

  *) usage_short
     exit 99
     ;;
  esac
done

if   [ -z "${queue_name}" ];then
     usage_short
     ${PRINT_CMD} "Missing [-q] option, please specify a queue name."
     exit 99
fi

#-- Check for virtual queue
pdebug "\nStep 1: Verify the existance of the virtual queue file.\n"
pdebug "---------------------------------------------------------\n"

if [ ! -f "${VIR_QUEUES_DIR}/${queue_name}/.${queue_name}" ];then
   ${PRINT_CMD} "Error: Printer queue [ ${queue_name} ] does not exist or the queue"
   ${PRINT_CMD} "is not defined by ${APPLICATION_NAME}."
   exit 1
fi

pdebug "Virtual queue found.\n"
pdebug "FILE=${VIR_QUEUES_DIR}/${queue_name}/.${queue_name}\n"
pdebug "..Continue..\n\n"


#-- Check for system queue
pdebug "\nStep 2: Verify the printer queue exists on the system.\n"
pdebug "--------------------------------------------------------\n"

cqs_out=`check_queue_status ${queue_name} 2>&1`
if [ "${?}" != "0" ];then
   if [ "${DEBUG}" = "1" ];then
      ${PRINTF_CMD} "Check Queue Status Output\n" >&2
      ${PRINTF_CMD} "-------------------------\n" >&2
      ${PRINTF_CMD} "${cqs_out}\n" >&2
      ${PRINTF_CMD} "-------------------------\n" >&2
   fi
   ${PRINT_CMD} "Error: Printer queue [ ${queue_name} ] does not exist."
   exit 1
fi
if [ "${DEBUG}" = "1" ];then
   ${PRINTF_CMD} "Check Queue Status Output\n" >&2
   ${PRINTF_CMD} "-------------------------\n" >&2
   ${PRINTF_CMD} "${cqs_out}\n" >&2
   ${PRINTF_CMD} "-------------------------\n" >&2
fi

pdebug "Printer queue [ ${queue_name} ] found.\n"
pdebug "..Continue..\n\n"


#-- Verify the queue is idle
pdebug "\nStep 3: Verify the printer queue is idle.\n"
pdebug "-------------------------------------------\n"

lp_status_output=`${LPC_CMD} status ${queue_name}`
if [ "${?}" != "0" ];then
   ${PRINT_CMD} "Error: ${LPC_CMD} command failed."
   exit 9
fi

${LPD_CMD} -V 2>&1 | ${GREP_CMD} "LPRng" >/dev/null 2>&1
if [ "${?}" = "0" ];then
   LPRNG="true"
   pdebug "LPD SubSystem is [ LPRng ]\n"
   ${PRINTF_CMD} "${lp_status_output}" | ${SED_CMD} -n 2p | ${GREP_CMD} " 0 " >/dev/null 2>&1
   status_return_code="$?"
else
   LPRNG="false"
   pdebug "LPD SubSystem is [ LPR-OLD ]\n"
   ${PRINTF_CMD} "${lp_status_output}" | ${SED_CMD} -n 4p | ${GREP_CMD} "no entries" >/dev/null 2>&1
   status_return_code="$?"
fi

if [ "${status_return_code}" != "0" ];then
   if [ ${DEBUG} = "1" ];then
      ${PRINTF_CMD} "Status command output is : \n${lp_status_output}\n" >&2
      ${PRINTF_CMD} "Printer Queue is not empty.\n" >&2
   fi
   ${PRINT_CMD} "There are jobs currently queued for printer queue ${queue_name}."
   ${PRINT_CMD} "Printer queues must be idle before they can be removed."
   exit 2
fi

if [ "${LPRNG}" = "true" ];then
   ${LPC_CMD} down ${queue_name} >/dev/null 2>&1
else 
   ${LPC_CMD} down ${queue_name} "The printer queue is down and will be removed immediately." >/dev/null 2>&1
fi

if [ "${?}" != "0" ];then
   ${PRINT_CMD} "Error: ${LPC_CMD} down {queue_name} command failed."
   exit 9
fi

pdebug "Queue is idle and down,  lets continue ..\n\n"
pdebug "..Continue..\n\n"


#--
pdebug "\nStep 3: Remove the printcap entry.\n"

#-- Some systems use a printcap.local file, if the
#-- file exists, our entry should be located in that
#-- file as well.
if [ -f "${PRINTCAP_LOCAL_FILE}" ];then
   pdebug "Removing entry from ${PRINTCAP_LOCAL_FILE}.\n"
   remove_printcap_entry ${PRINTCAP_LOCAL_FILE}
   pdebug "Removing entry from ${PRINTCAP_LOCAL_FILE} was successful.\n"
   pdebug "..Continue..\n\n"
fi

#-- Remove the main entry 
remove_printcap_entry ${PRINTCAP_FILE}

pdebug "Removing entry from ${PRINTCAP_FILE} was successful\n"
pdebug "..Continue..\n\n"


pdebug "\nStep 4: Let LPD reread the printcap file.\n"
pdebug "---------------------------------------------------\n"

${LPC_CMD} reread all >/dev/null 2>&1
if [ "${?}" = "0" ];then
   pdebug "lpc reread all completed.\n"
else
   pdebug "lpc reread all failed.\n"
fi

if [ ${completely_remove} = "true" ];then
   pdebug "Printer queue was removed ....\n"
   pdebug "Removing spool directory\n"
   cd ${VIR_QUEUES_DIR}
   ${RM_CMD} -rf ${queue_name}
   if [ "${?}" != "0" ];then
      ${PRINT_CMD} "Printer queue ${queue_name} was removed successfully."
      ${PRINT_CMD} "However, the spool directory was not removed.  You will"
      ${PRINT_CMD} "need to remove it manually."
   else
      ${PRINT_CMD} "Printer queue ${queue_name} was removed successfully."
   fi
else
   ${PRINT_CMD} "Printer queue ${queue_name} was removed successfully."
fi

exit 0
