#!/bin/sh

########################################################################
#             (c) Copyright 2003 Lexmark International, Inc.           #
#                        All rights reserved                           #
########################################################################
#                                                                       
#  This command is used to manage printer classes and enable/diable,
#  accept/reject print queues.
#                                                                       
########################################################################
#                                                                       
#  0 --- Success  -  Command completed successfully
#  1 --- Failed   -  Command failed.                                                                   
#  2 --- Failed   -  Command failed.
# 99 --- Error    -  Command line options incorrect
#
########################################################################

#-- set debug to false by default. To turn debug "on" set to "1"
DEBUG=0

#-- set initial return_code
return_code=99

#-- initialize variables to 0
change_class="0"
change_status="0"
run_command_line="true"


PRODUCT_LINK="/usr/genprint"
SCRIPTS_DIR="${PRODUCT_LINK}/bin/.scripts"

#-- load the shared script
. ${SCRIPTS_DIR}/shared

#-- initialize variables:
initialize user

program_name=`${BASENAME_CMD} $0`

#-- usage_short text
usage_short ()
{
   ${PRINT_CMD}
   ${PRINT_CMD} "USAGE:"
   ${PRINT_CMD} "    ${program_name} -q queue -m enable|disable|accept|reject [-x] ${SARG}"
   if [ "${PRT_SUBSYSTEM}" = "SYSTEM_V" ];then
      ${PRINT_CMD} "    ${program_name} -q queue -c -a class1,class2,... ${SARG} [-x]"
      ${PRINT_CMD} "    ${program_name} -q queue -c -r class1,class2,... ${SARG} [-x]"
   fi
   ${PRINT_CMD} "    ${program_name} -h"
   ${PRINT_CMD} "    ${program_name} -v" 
   ${PRINT_CMD}
   ${PRINT_CMD} "Try '${program_name} -h' for more information"
   ${PRINT_CMD}
}

#-- usage_long text
usage_long ()
{
if [ "${PRT_SUBSYSTEM}" = "BSD" ];then
${CAT_CMD} << EOT

${program_name}
    Modifies a print queue's status.

USAGE:
    ${program_name} -q queue -m enable|disable|accept|reject [-x]
    ${program_name} -h
    ${program_name} -v

OPTIONS:
    -q queue
       Specify the printer queue name.

    -m enable|disable|accept|reject
       Specify the state to apply to the queue:
         enable:   The normal state of operation where new and existing print
                   jobs in the queue are actively processed.
         disable:  Halt the processing of new and existing print jobs in the
                   queue.  New jobs can be successfully submitted, but will not
                   be processed until the queue is enabled.
         accept:   Allow the submission of new print jobs to the queue.
         reject:   Disallow the submission of new print jobs to the queue.

    -x
       Run ${program_name} in debug mode.

    -h
       Display this help and exit.

    -v
       Display version information and exit.


EXAMPLES:
    Change the status of the printer queue myqueue to reject print jobs:

       $ ${program_name} -q myqueue -m reject

EOT
elif [ "${PRT_SUBSYSTEM}" = "SYSTEM_V" ];then
${CAT_CMD} << EOT

${program_name}
    Modifies a print queue's status and printer class associations.

USAGE:
    ${program_name} -q queue -m enable|disable|accept|reject [-x] ${SARG}
    ${program_name} -q queue -c -a class1,class2,... [-x] ${SARG}
    ${program_name} -q queue -c -r class1,class2,... [-x] ${SARG}
    ${program_name} -h
    ${program_name} -v

OPTIONS:
    -q queue
       Specify the printer queue name.

    -m enable|disable|accept|reject
       Specify the state to apply to the queue:
         enable:   The normal state of operation where new and existing print
                   jobs in the queue are actively processed.
         disable:  Halt the processing of new and existing print jobs in the
                   queue.  New jobs can be successfully submitted, but will not
                   be processed until the queue is enabled.
         accept:   Allow the submission of new print jobs to the queue.
         reject:   Disallow the submission of new print jobs to the queue.

    -c
       Change printer queue's class association.

    -a class1,class2,...
       Specify any number of class names seperated by a comma or space.  This
       option is used to add a printer queue to a printer class.

    -r class1,class2,...
       Specify any number of class names seperated by a comma or space.  This
       option is used to remove a printer queue from a printer class.

    -x
       Run ${program_name} in debug mode.

    -h
       Display this help and exit.
    -v
       Display version information and exit.

EXAMPLES:
    Change the status of the printer queue myqueue to reject print jobs:

       $ ${program_name} -q myqueue -m reject

    Add the printer queue myqueue to printer classes class1 and class2:

       $ ${program_name} -q myqueue -c -a class1,class2

    Add the printer queue myqueue to printer classes class1 and class2 and
    remove it from printer class class9:

       $ ${program_name} -q myqueue -c -a class1,class2 -r class9

EOT
fi
}

#--------------------------------------------------
#--  This function is called to associate a printer 
#--  queue with a printer class. 
#-- 
#--  Syntax:
#--  add_class <queue_name> <class_list>
#--
#--  Return 0 : Success
#--  Return 1 : Failed
#--
#--
add_classes()
{
   lqName="${1}" 
   lclassList="${2}"
   return_code=1

   for class in `${PRINT_CMD} ${lclassList} | ${SED_CMD} -e 's/,/ /g'`
   do
      check_characters "${class}" "Class"
      if [ "${?}" != "0" ];then
         ${PRINT_CMD} "Associating printer queue $lqName to printer class ${class} failed!"
         return_code=1
         continue
      fi

      #-- Add printer queue to printer class
      pdebug "${LPADMIN_CMD} -p${lqName} -c${class} ... "
      out=`${LPADMIN_CMD} -p${lqName} -c${class} 2>&1`
      if [ "${?}" != "0" ];then
	 ${PRINT_CMD} "Associating printer queue $lqName to printer class ${class} failed!"
         pdebug "COMMAND OUT:\n$out\n"
         return_code=1
      else
         #-- Set printer class to accepting print requests
         RC="0"
	 ${ACCEPT_CMD} "${class}" >/dev/null
	 ${ENABLE_CMD} "${class}" >/dev/null
         if [ "${RC}" = "0" ];then
	    ${PRINT_CMD} "Printer queue $lqName is now associated to printer class ${class}."
            return_code=0
         else
	    ${PRINT_CMD} "Printer queue $lqName is now associated to printer class ${class}."
	    ${PRINT_CMD} "However the printer class ${class} is not accepting requests."
            return_code=2
         fi
      fi 
   done
   return ${return_code}
}


#----------------------------------------------
#--  This function is called to disassociate a
#--  printer queue from a printer class. 
#-- 
#--  Syntax:
#--  remove_class <queue_name> <class_list>
#--
#--  Return 0 : Success
#--  Return 1 : Failed
#--
#--
remove_classes()
{
   lqName="${1}" 
   lclassList="${2}"
   return_code="1"

   for class in `${PRINT_CMD} ${lclassList} | ${SED_CMD} -e 's/,/ /g'`
   do
      pdebug "${LPADMIN_CMD} -p${lqName} -r${class} ... "
      out=`${LPADMIN_CMD} -p${lqName} -r${class} 2>&1`
      if [ "${?}" != "0" ];then
	 ${PRINT_CMD} "Disassociating printer queue $lqName from printer class ${class} failed!"
         pdebug "COMMAND OUTPUT:\n$out\n"
         return_code=1
      else
	 ${PRINT_CMD} "Printer queue $lqName is now disassociated from printer class ${class}."
         return_code=0
      fi 
   done
   return ${return_code}
}


#-- Check for Admin user
is_privileged_user
if [ "${?}" != "0" ];then
   ${PRINT_CMD}
   ${PRINT_CMD} "You must be an administrator to run this program."
   ${PRINT_CMD}
   exit 99
fi

#-- re-initialize variables for admin mode:
initialize admin

#-- MAIN --#
#-- parse command line arguments --#

check_subsystem()
{
   if [ "${PRT_SUBSYSTEM}" = "BSD" ];then
      usage_short
      exit 99
   fi
}

while getopts Ga:cm:q:r:xsvh arg
do
  case "${arg}" in
    G) run_command_line="false"
		 ASKSHUTDOWN="no"
       ;;

    a) check_subsystem
       add_class_list="${OPTARG}"
       ;;

    c) check_subsystem
       change_class="1"
       ;;

    r) check_subsystem
       remove_class_list="${OPTARG}"
       ;;

    h) usage_long
       exit 99
       ;;

    m) change_status="1"
       status_command="${OPTARG}"
       ;;

    q) queue_name="${OPTARG}"  
       ;;

    s) ASKSHUTDOWN="no"
       ;;

    v) display_version
       exit 99
       ;;

    x) DEBUG=1;;

    *) usage_short
       exit 99;;
  esac 
done

#-- verify a queue name was specified
if [ -z "${queue_name}" ];then 
   usage_short
   ${PRINT_CMD} "Missing [-q] option, please specify a queue name."
   exit 99
fi

#-- check for mutually exclusive command line arguments --#
if [ "${change_class}" = "1" -a "${change_status}" = "1" ]; then
   usage_short
   option_mutually_exclusive "-c|-m"
   exit 99
fi 

#-- verify the queue is an existing system queue
if [ ! -f "${VIR_QUEUES_DIR}/${queue_name}/.${queue_name}" ];then
   ${PRINT_CMD} "Printer queue ${queue_name} does not exist."
   exit 1 
fi
   
check_for_system_queue "${queue_name}"
if [ "$?" != "0" ];then
   ${PRINT_CMD} "Printer queue ${queue_name} does not exist."
   exit 1 
fi


if [ "${change_class}" = "1" ];then


   #-- check to make sure either the -a or -r flag was given
   if [ -z "${add_class_list}" -a -z "${remove_class_list}" ];then 
      usage_short
      ${PRINT_CMD} "The [-c] flag requires either the -a or -r options."
      exit 99
   fi

   #-- associated the printer queue with the printer class list
   if [ ! -z "${add_class_list}" ];then
         add_classes "${queue_name}" "${add_class_list}"
   fi
   add_class_rc="$?"

   #-- disassociated the printer queue from the printer class list
   if [ ! -z "${remove_class_list}" ];then
      remove_classes "${queue_name}" "${remove_class_list}"
   fi
   remove_class_rc="${?}"

   #-- check return codes 
   if [ "${add_class_rc}" = "1" -o "${remove_class_rc}" = "1" ];then
      exit_code="1"
   else
      exit_code="0"
   fi 


   exit ${exit_code}

elif [ "${change_status}" = "1" ];then

    case "${status_command}" in
      accept)  accept_queue  ${queue_name}  ;;
      reject)  reject_queue  ${queue_name}  ;;
      enable)  enable_queue  ${queue_name}  ;;
      disable) disable_queue ${queue_name}  ;;
	    *) usage_short
               ${PRINT_CMD} "The [-m] argument is invalid."
	       exit 99;;

    esac
    exit ${?} 

else
   #-- missing -m or -c flag
   usage_short
   exit 99
fi

