#!/bin/sh

########################################################################
#             (c) Copyright 2002 Lexmark International, Inc.           #
#                        All rights reserved                           #
########################################################################
#                                                                       
# This command is used to list virtual & physical device information.
#
########################################################################
#
# Exit codes:
#
#  0  --- Success - Virtual Device found and/or script ran successfully
#  1  --- Error   - Virtual Device not found
# 99  --- Error   - Command line options incorrect
#
########################################################################

#-- set debug to false by default. To turn debug "on" set to "1"
DEBUG=0

#-- set initial return_code
return_code=99

#-- initialize variables to 0
ls_physical_devices=0
ls_a_virtual_device=0
ls_virtual_devices=0
long_listing="true"


PRODUCT_LINK="/usr/genprint"
SCRIPTS_DIR="${PRODUCT_LINK}/bin/.scripts"

#-- load the shared script
. ${SCRIPTS_DIR}/shared

#-- initialize
initialize

program_name=`${BASENAME_CMD} $0`

#-- usage_short text
usage_short ()
{
${CAT_CMD} << EOT

USAGE:
    ${program_name} -d device [-x]
    ${program_name} -l [-x]
    ${program_name} -p [-x]
    ${program_name} -h
    ${program_name} -v

Try '${program_name} -h' for more information

EOT
}

#-- usage_long text
usage_long ()
{
${CAT_CMD} << EOT

${program_name}
    Lists virtual and physical device information.

USAGE:
    ${program_name} -d device [-x]
    ${program_name} -l [-x]
    ${program_name} -p [-x]
    ${program_name} -h
    ${program_name} -v

OPTIONS:
    -d device
       List detailed information regarding a specific virtual device.

    -l
       List detailed information regarding all virtual devices.

    -p
       List commonly physical devices for this system.

    -x
       Run ${program_name} in debug mode.

    -h
       Display this help and exit.

    -v
       Display version information and exit.

EXAMPLES:
    List information about the device mydevice:

       $ ${program_name} -d mydevice

    List information about all virtual devices:

       $ ${program_name} -l

    List possible printer devices on this system:

       $ ${program_name} -p

EOT
}

#---------------------------------------------------
#--  This function is called to check and list known
#--  physical devices.
#--
#--  Syntax:
#--  check_device_group device_paths device_type
#--

#--
check_device_group()
{
   device_list="${1}"
   device_type="${2}"

   if [ "${device_list}" = "UNAVAILABLE" ];then
       pdebug "\n${device_type} devices are unsupported.\n"
       return
   fi

   pdebug "\nChecking for ${device_type} devices.\n"
   pdebug "[ ${device_list} ]\n"

   if [ "${device_type}" = "bpp" ];then
      #-- set bpp enabled true
      ${PRINT_CMD} "bpp#available#0"
      #-- set device_type to parallel for the remainder of this function
      device_type="parallel"
      lbpp="1"
   else
      #-- set bpp enabled false
      lbpp="0"
   fi

   #-- convert the colon seperated device_list variable to
   #-- a space separated variable.
   device_list="`${PRINT_CMD} ${device_list} | ${SED_CMD} -e 's/:/ /g'`"

   #-- check for each device path
   for device in ${device_list}
   do
      pdebug "\nChecking for device [ ${device} ]"
      if [ -c "${device}" ]; then
         pdebug ".. found.\n"
         ${PRINT_CMD} "${device_type}#${device}#${lbpp}"
      else
         pdebug ".. not found. --SKIPPING--\n"
      fi
   done

   #-- always include an other device except for bpp type.
   if [ "${lbpp}" = "0" ];then
      ${PRINT_CMD} "${device_type}#other#0"
   fi

   unset device_list
   unset device_type
   unset lbpp
}


#-- MAIN --#
#-- parse command line arguments --#
while getopts Gd:lpvxh? arg
do
  case "${arg}" in
  G) long_listing="false"
     ;; 

  d) virtual_device="${OPTARG}"
     ls_a_virtual_device=1
     #-- check for mutually exclusive command line arguments 
     if [ "${ls_physical_devices}" = "1" -o "${ls_virtual_devices}" = "1" ]; then
        usage_short
        option_mutually_exclusive "-d|-l|-p"
        exit 99
     fi
     ;;

  h) usage_long
     exit 99
     ;;

  l) ls_virtual_devices=1
     #-- check for mutually exclusive command line arguments 
     if [ "${ls_physical_devices}" = "1" -o "${ls_a_virtual_device}" = "1" ]; then
        usage_short
        option_mutually_exclusive "-d|-l|-p"
        exit 99
     fi;;

     
  p) ls_physical_devices=1
     #-- check for mutually exclusive command line arguments 
     if [ "${ls_virtual_devices}" = "1" -o "${ls_a_virtual_device}" = "1" ]; then
        usage_short
        option_mutually_exclusive "-d|-l|-p"
        exit 99
     fi
     ;;
 
  v) display_version
     exit 99
     ;;

  x) DEBUG=1
     ;;

  *) usage_short
     exit 99
     ;;

  esac 
done

if [ "${ls_physical_devices}" = "1" ]; then
   #-- List Physcial Devices 

   check_device_group "${BPP_PARALLEL_DEVS}" bpp
   check_device_group "${PARALLEL_DEVS}" parallel
   check_device_group "${SERIAL_DEVS}" serial
   check_device_group "${USB_DEVS}" usb

elif [ "${ls_virtual_devices}" = "1" ]; then
    #-- List all virtual device information 

    if [ ! -d "${VIR_DEVICES_DIR}" ];then
       ${PRINT_CMD} "ERROR:  The virtual device directory is missing."
       ${PRINT_CMD}
       ${PRINT_CMD} "Run the [ setup.${PRODUCT_NAME} ] script to recreate the appropriate"
       ${PRINT_CMD} "application directories."
       ${PRINT_CMD}
       exit 2 
    fi

    devices=`${LS_CMD} ${VIR_DEVICES_DIR} 2>/dev/null`
    if [ "${?}" != "0" ];then 
       ${PRINT_CMD} "ERROR: ${VIR_DEVICES_DIR} directory not found."
       exit 1
    else
       get_device_settings "${devices}"
    fi
       
elif [ "${ls_a_virtual_device}" = "1" ]; then

    get_device_settings "${virtual_device}"
    rc="${?}"
    if   [ "${rc}" = "0" ];then
         pdebug "Virtual Device [ ${virtual_device} ] found.\n"
         exit 0
    elif [ "${rc}" = "1" ];then
         ${PRINT_CMD} "Virtual Device [ ${virtual_device} ] does not exist."
         exit 1
    else
         ${PRINT_CMD} "Virtual Device [ ${virtual_device} ] is corrupt."
         exit 1
    fi
else
    usage_short
    exit 99
fi

exit 0
