#!/bin/sh

########################################################################
#             (c) Copyright 2002 Lexmark International, Inc.           #
#                        All rights reserved                           #
########################################################################
#
# This command provides the system type, classses, printer queues, and
# virtual queues. 
# 
#
########################################################################
#
# Exit codes:
# 
#  0  --- Success -  Printer Queue exists.
#  1  --- Success -  Printer Queue exists not exist.
#  9  --- Error   -  Invalid device name
# 99  --- Error   -  Usage Error
# 
# OS Types [ -t ] Flag 
# --------------------
#  1  --- Print Subsystem supports Classes
#  2  --- Print Subsystem supports Classes, plus prompt to shutdown lpsched
#  3  --- Print Subsystem supports Aliases
#
########################################################################

#-- set debug to false by default. To turn debug "on" set to "1"
DEBUG=0

#-- initialize variables to 0
list_queue="0"
list_type="0"
check_for_queue="0"
list_alias_or_class="0"

run_command_mode="true"


PRODUCT_LINK="/usr/genprint"
SCRIPTS_DIR="${PRODUCT_LINK}/bin/.scripts"

#-- load the shared script
. ${SCRIPTS_DIR}/shared

#-- initialize
initialize

program_name=`${BASENAME_CMD} $0`

#-- usage_short text
usage_short ()
{ 
   ${PRINT_CMD}
   ${PRINT_CMD} "USAGE:"
   if   [ "${PRT_SUBSYSTEM}" = "BSD" ];then
        ${PRINT_CMD} "    ${program_name} -q queue -a [-x]"
   elif [ "${PRT_SUBSYSTEM}" = "SYSTEM_V" ];then 
        ${PRINT_CMD} "    ${program_name} -a [-x]"
   fi
   ${PRINT_CMD} "    ${program_name} -q queue -c system|virtual [-x]"
   ${PRINT_CMD} "    ${program_name} -l [-u directory] [-x]"
   ${PRINT_CMD} "    ${program_name} -t [-x]"
   ${PRINT_CMD} "    ${program_name} -v"
   ${PRINT_CMD} "    ${program_name} -h"
   ${PRINT_CMD}
   ${PRINT_CMD} "Try '${program_name} -h' for more information"
   ${PRINT_CMD}
}

#-- usage_long text
usage_long ()
{
if   [ "${PRT_SUBSYSTEM}" = "BSD" ];then
     ul_help="printer aliases"
elif [ "${PRT_SUBSYSTEM}" = "SYSTEM_V" ];then
     ul_help="printer classes"
fi

${CAT_CMD} << EOT

${program_name}
    Provides information regarding the print subystem, ${ul_help}, virtual 
    queues and system queues. 

USAGE:
EOT

if   [ "${PRT_SUBSYSTEM}" = "BSD" ];then
     ${PRINT_CMD} "    ${program_name} -q queue -a [-x]"
elif [ "${PRT_SUBSYSTEM}" = "SYSTEM_V" ];then
     ${PRINT_CMD} "    ${program_name} -a [-x]"
fi
${CAT_CMD} << EOT
    ${program_name} -q queue -c system|virtual [-x]
    ${program_name} -l [-u directory] [-x]
    ${program_name} -t [-x]
    ${program_name} -v
    ${program_name} -h

OPTIONS:
    -q queue
       Specify the printer queue name.

EOT
  
if   [ "${PRT_SUBSYSTEM}" = "BSD" ];then
     ${PRINT_CMD} "    -a"
     ${PRINT_CMD} "       List printer aliases."
elif [ "${PRT_SUBSYSTEM}" = "SYSTEM_V" ];then
     ${PRINT_CMD} "    -a"
     ${PRINT_CMD} "       List printer classes."
fi
${CAT_CMD} << EOT

    -c system|virtual
       Specify the type of queue to check.  The 'system' parameter checks for 
       all queues defined in the system.  The 'virtual' parameter only checks 
       queues defined by ${APPLICATION_NAME}.

    -l
       List detailed information regarding each printer queue. This information
       includes the name, printer model, description, device name, emulation 
       and status of each queue.

    -t
       Return the print subsystem type.

    -u dir
       Specify the current user's directory.

    -x
       Run ${program_name} in debug mode.

    -h
       Display this help and exit.

    -v
       Display version information and exit.

EXAMPLES:

EOT
if   [ "${PRT_SUBSYSTEM}" = "BSD" ];then
     ${PRINT_CMD} "    List printer aliases for the queue myqueue:"
     ${PRINT_CMD} "      $ ${program_name} -q myqueue -a"
elif [ "${PRT_SUBSYSTEM}" = "SYSTEM_V" ];then
     ${PRINT_CMD} "    List all printer classes:"
     ${PRINT_CMD} "      $ ${program_name} -a"
fi
${CAT_CMD} << EOT

    Check if myqueue exists in the system:

       $ ${program_name} -q myqueue -c system

    Check if myqueue was defined by ${APPLICATION_NAME}:

       $ ${program_name} -q myqueue -c virtual

    List all queues defined by ${APPLICATION_NAME}:

       $ ${program_name} -l

EOT
}


#---------------------------------------------------
#--  This function is called to check for a system
#--  queue. 
#-- 
#--  Syntax:
#--  chkqueue <queue_name>
#--
#--  Return 0 -- Queue Found   
#--  Return 1 -- Queue Not Found   
#--
chkqueue()
{
   lqName=${1}
 
   pdebug "Checking for [ ${lqName} ]\n"
   queue_status=`check_queue_status ${lqName}`
   RC="${?}"
   if   [ "${RC}" = "0" ];then
        pdebug "${queue_status}\n"
        ${PRINTF_CMD} "Printer Queue [ ${lqName} ] was found.\n"
        return 0
   elif [ "${RC}" = "2" ];then
        pdebug "${queue_status}\n"
        ${PRINTF_CMD} "Printer Class [ ${lqName} ] was found.\n"
        return 0
   else
        pdebug "${queue_status}\n"
        ${PRINTF_CMD} "Printer Queue [ ${lqName} ] was not found.\n"
        return 1
   fi
}



#---------------------------------------------------
#--  This function is called to return detailed
#--  information regarding printer queues.
#--
#--  Syntax:
#--  list_valid_queues 
#--  list_valid_queues <queue_name>
#--
list_valid_queues()
{
   return_code="1"

   #-- If $1 is NULL, show all queues
   if [ -z "${1}" ];then
      pdebug "Checking all ${APPLICATION_NAME} defined printer queues.\n"
      queues=`${LS_CMD} ${VIR_QUEUES_DIR} 2>/dev/null`
      if [ "${?}" != "0" ];then
         pdebug "Directory: ${VIR_QUEUES_DIR} does not exist.\n"
         ${PRINTF_CMD} "No printer queues were found.\n"
         exit 1
      fi
   else
      lqName="${1}"
      pdebug "Lets Check a specific queue.\n"
      queues=`${LS_CMD} -d ${VIR_QUEUES_DIR}/${lqName} 2>/dev/null`
      if [ "${?}" != "0" ];then
         pdebug "File: ${VIR_QUEUES_DIR}/${lqName} does not exist.\n"
         ${PRINTF_CMD} "Printer Queue [ ${lqName} ] was not found.\n"
         exit 1 
      fi
      queues=`${BASENAME_CMD} ${queues}`
   fi
   
   for queue in `${PRINT_CMD} ${queues}`
   do
      user_has_settings="false"
      pdebug "Checking for User settings.\n"
      if [ ! -z "${user_home}" ]; then 
         queue_settings=`get_queue_settings ${user_home}/vir_queues/${queue}/.${queue}`
         RC="${?}"
         if [ "${RC}" = "0" ];then
            pdebug "User has printer queue settings.\n"
            user_has_settings="true"
         fi
      fi
      if [ "${user_has_settings}" = "false" ]; then 
            pdebug "User Setting not Found, Fetching System Settings\n"
            queue_settings=`get_queue_settings ${VIR_QUEUES_DIR}/${queue}/.${queue}`
            RC="${?}"
      fi
      if [ "${RC}" != "0" ];then
         pdebug "Print Queue [ ${queue} ] is not defined by ${APPLICATION_NAME}.\n"
         return_code="1"
      else 
         queue_status=`check_queue_status ${queue}`
         if [ "${?}" != "0" ];then
            pdebug "Print Queue [ ${queue} ] is not a system queue\n\n"
            return_code="1"
         else 
            if [ "${run_command_mode}" = "false" ];then
               ${PRINT_CMD} "${queue_settings}:${queue_status}:${user_has_settings}"
            else
               ${PRINTF_CMD} "Queue Name           : "
               ${PRINT_CMD} ${queue_settings} | ( IFS=":" read a b c d e f; ${PRINT_CMD} ${a}) 
               ${PRINTF_CMD} "Printer Model        : "
               ${PRINT_CMD} ${queue_settings} | ( IFS=":" read a b c d e f; ${PRINT_CMD} ${b}) 
               ${PRINTF_CMD} "Description          : "
               ${PRINT_CMD} ${queue_settings} | ( IFS=":" read a b c d e f; ${PRINT_CMD} ${c}) 
               ${PRINTF_CMD} "Virtual Device       : "
               ${PRINT_CMD} ${queue_settings} | ( IFS=":" read a b c d e f; ${PRINT_CMD} ${d}) 
               ${PRINTF_CMD} "Queue Emulation      : "
               ${PRINT_CMD} ${queue_settings} | ( IFS=":" read a b c d e f; ${PRINT_CMD} ${e}) 
               ${PRINTF_CMD} "Printer Queue Status : "
               ${PRINT_CMD} ${queue_status} | ( IFS=":" read a b; ${PRINT_CMD} ${a}) 
               ${PRINTF_CMD} "Printer Queue Status : "
               ${PRINT_CMD} ${queue_status} | ( IFS=":" read a b; ${PRINT_CMD} ${b}) 
               ${PRINT_CMD}
            fi
            return_code="0"
         fi
      fi
   done
   return ${return_code}
}

#---------------------------------------------------
#--  This function is called to check to see if the
#--  options specified on the command line are 
#--  mutually exclusive 
#-- 
#--  Syntax:
#--
#--  check_option_list
#--

check_option_list()
{
  option_count=`${EXPR_CMD} ${list_queue} + ${list_type} + ${check_for_queue} + ${list_alias_or_class}`

  if [ ${option_count} -gt 1 ];then
     usage_short
     ${PRINT_CMD} "Specified options are mutually exclusive."
     exit 99
  fi
}


#-- MAIN --#
#-- parse command line arguments --#
while getopts Gac:q:lptu:vxh? arg
do
  case "${arg}" in
  G) run_command_mode="false"
     ;;
 
  a) list_alias_or_class=1
     check_option_list
     ;;

  c) check_type=${OPTARG}
     check_for_queue=1
     if [ "${check_type}" != "system" -a "${check_type}" != "virtual" ];then
        usage_short
        ${PRINT_CMD} "The [ -c ] option is invalid.  Specify [ system or virtual ] ."
        exit 99
     fi
     check_option_list
     ;;
  
  h) usage_long
     exit 99
     ;;
 
  l) list_queue=1
     check_option_list
     ;;

  p) 
     JPID=`${PS_CMD} | ${GREP_CMD} [j]ava`
     if [ "${?}" != "0" ];then
        ${PRINT_CMD} "Java Process ID not found."
        exit_code=1
     else
        ${PRINTF_CMD} "${JPID}\n" | ${TAIL_CMD} -1 | ${AWK_CMD} '{print $1}' | ${SED_CMD} -e 's/ //g'
        exit_code=0
     fi
     exit ${exit_code}
     ;;
 
  q) queue_name=${OPTARG}
     check_characters ${queue_name} "Queue"
     if [ "${?}" != "0" ];then
        exit 9
     fi
     ;;

  h) usage_long

     ;;

  t) list_type=1
     check_option_list
     ;;

  v) display_version
     exit 99
     ;;

  u) user_home="${OPTARG}"
     ;;

  x) DEBUG=1
     ;;

  *) usage_short
     exit 99
     ;;
  esac 
done

#-- Check to see if regular users have access to the spool directories.
check_spool_dir_permissions

#-- Flag -a : List all aliases or classes
if   [ "${list_alias_or_class}" = "1" ];then
     if   [ "${PRT_SUBSYSTEM}" = "BSD" ];then
	   if [ -z "${queue_name}" ];then
	      usage_short
	      ${PRINT_CMD} "Missing [-q] option, please specify a queue name."
	      exit 99
	   else
	      #-- verify the queue is an existing system queue
	      if [ ! -f "${VIR_QUEUES_DIR}/${queue_name}/.${queue_name}" ];then
		 ${PRINT_CMD} "Printer queue ${queue_name} does not exist."
		 exit 1
	      fi

	      chkqueue ${queue_name} >/dev/null 2>&1
	      if [ "${?}" = "1" ];then
		 ${PRINT_CMD} "Printer Queue [ ${queue_name} ] was not found"
		 exit 1
	      fi
	   fi
	   if [ "${run_command_mode}" = "false" ];then
	      ${CAT_CMD} ${PRINTCAP_FILE} | ${EGREP_CMD} -E "^${queue_name}" | \
					    ${SED_CMD} -e 's/:\\//g' | ( IFS='|'; read a b; ${PRINT_CMD} $b)
	   else
	      ${PRINTF_CMD} "\nPrinter Queue : ${queue_name}\n"
	      ${PRINTF_CMD} "Aliases       : " 
	      ${CAT_CMD} ${PRINTCAP_FILE} | ${EGREP_CMD} -E "^${queue_name}" | ${SED_CMD} 's/:\\//g' | \
				 ( IFS='|'; read a b; ${PRINT_CMD} $b ) | ${SED_CMD} -e 's/ /,/g'
	      ${PRINT_CMD}
	   fi
	   exit 0
elif [ "${PRT_SUBSYSTEM}" = "SYSTEM_V" ];then
        if [ -d ${CLASSES_DIR} ];then 
	   classes=`${LS_CMD} ${CLASSES_DIR} 2>/dev/null`
	   is_cmd_success "$?" "0" "yes" "ERROR: ${CLASSES_DIR} directory not found." \
			 "${LS_CMD} ${CLASSES_DIR}" "1" "no"
	   for class in `${PRINT_CMD} ${classes}`
	   do
	       if [ "${run_command_mode}" = "true" ];then
		  ${PRINT_CMD} "Printer Class     :  ${class}"
		  ${PRINTF_CMD} "Associated Queues : "
		  ${AWK_CMD} ' {printf(" %s",$1)}' ${CLASSES_DIR}/${class}
		  ${PRINTF_CMD} "\n\n"
	       else
		  ${PRINTF_CMD} "${class}"
		  ${AWK_CMD} ' {printf(":%s",$1)}' ${CLASSES_DIR}/${class}
		  ${PRINTF_CMD} "\n"
	       fi
	   done
        elif [ -f /etc/cups/classes.conf ];then
          ${CAT_CMD} /etc/cups/classes.conf | ${GREP_CMD} -v "^#" | \
          ${SED_CMD} -e 's/<Class \(.*[^>]\)>/ClassName \1/g'     | \
          ${AWK_CMD} '
		     /ClassName/  { printf("%s",$2)  }
		     /Printer/    { printf(":%s",$2) }
		     /<\/Class>/  { printf("\n")    }
                     '
        fi
        exit 0
fi

#-- Flag -l : List all virtual queues 
elif [ "${list_queue}" = "1" ];then
     list_valid_queues
     exit 0;

#-- Flag -t : List Print Subsystem type
elif [ "${list_type}" = "1" ];then
     get_os_type
     exit ${?}

#-- Flag -c : Check a printer queue
elif [ "${check_for_queue}" = "1" ];then
     if [ -z "${queue_name}" ];then
        usage_short
        ${PRINT_CMD} "Missing [-q] option, please specify a queue name."
        exit 99
     fi
     if [ "${check_type}" = "virtual" ];then
        list_valid_queues ${queue_name}
        exit ${?}
     else
        chkqueue ${queue_name}
        exit ${?}
     fi

#-- No flag specified.
else
     usage_short
     ${PRINT_CMD} "ERROR: An option is required."
     exit 99
fi

