#!/bin/sh

########################################################################
#             (c) Copyright 2002 Lexmark International, Inc.           #
#                        All rights reserved                           #
########################################################################
#                                                                       
# This command is removes virtual devices information.
#
########################################################################
#
# Exit codes:
#
#  0  --- Success - Virtual Device removed
#  1  --- Error   - Virtual Device does not exist
#  2  --- Error   - Virtual Device was not removed, queues associated
#  3  --- Error   - Virtual Device was not removed, removing device file failed
# 99  --- Error   - Command line options incorrect
#
########################################################################

#-- set debug to false by default. To turn debug "on" set to "1"
DEBUG=0

#-- set initial variables
run_command_line="true"
remove_device="true"


PRODUCT_LINK="/usr/genprint"
SCRIPTS_DIR="${PRODUCT_LINK}/bin/.scripts"

#-- load the shared script
. ${SCRIPTS_DIR}/shared

#-- initialize
initialize

program_name="rmdevice"

#-- usage_short text
usage_short ()
{
${CAT_CMD} << EOT

USAGE:
    ${program_name} -d device [-x]
    ${program_name} -h
    ${program_name} -v

Try '${program_name} -h' for more information

EOT
}

#-- usage_long text
usage_long ()
{
${CAT_CMD} << EOT

${program_name}
    Removes a virtual device.

USAGE:
    ${program_name} -d device [-x]
    ${program_name} -h
    ${program_name} -v

OPTIONS:
    -d device
       Specify the virtual device name to remove.

    -x
       Run ${program_name} in debug mode.

    -h
       Display this help and exit.

    -v
       Display version information and exit.

EXAMPLES:
    Remove the device named mydevice:

       $ ${program_name} -d mydevice

EOT
}

#-- Check for Admin user
is_privileged_user
if [ "${?}" != "0" ];then
   ${PRINT_CMD}
   ${PRINT_CMD} "You must be an administrator to run this program."
   ${PRINT_CMD}
   exit 99
fi

#-- MAIN --#
#-- parse command line arguments --#
while getopts d:Gvxh? arg
do
  case "${arg}" in

  d) virtual_device="${OPTARG}"
     check_characters "${virtual_device}" "Device"
     if [ "${?}" != "0" ];then
        exit 99
     fi
     ;;
 
  G) run_command_line="false"
     ;;

  h) usage_long
     exit 99
     ;;

  v) display_version
     exit 99
     ;;

  x) DEBUG=1
     ;;

  *) usage_short
     exit 99
     ;;

  esac 
done

#-- verify that the virtual device name was specified
if [ -z "${virtual_device}" ];then
   usage_short
   ${PRINT_CMD} "Missing [-d] option, specify a virtual device name."
   exit 99
fi

#-- check to see if virtual device file exists
if [ ! -f "${VIR_DEVICES_DIR}/${virtual_device}" ];then
   ${PRINT_CMD} "Virtual Device [ ${virtual_device} ] does not exist."
   exit 1
fi

#-- check to see if the virtual device is currently associated
#-- with an existing print queue.

#-- grab all print queue names
queue_list="`${LS_CMD} ${VIR_QUEUES_DIR} 2>/dev/null`"

#-- check each queue
pdebug "\nChecking print queue device association.\n" 
for queue in ${queue_list}
do
    pdebug "Queue [ ${queue} ] is associated with device [ ${virtual_device} ] -- " 
    fullpath="${VIR_QUEUES_DIR}/${queue}/.${queue}"
    ${SED_CMD} 10q ${fullpath} 2>&1 | ${EGREP_CMD} "^device_name=${virtual_device}$" >/dev/null 2>&1
    if [ "${?}" = "0" ];then
       cqs_out=`check_queue_status ${queue} 2>&1`
       if [ "${?}" != "0" ];then
          #-- Queue does not exist on the system.
          real_queue="false"
       else
          real_queue="true"
       fi

       if [ "${real_queue}" = "true" ];then
          pdebug "TRUE\n" 
          remove_device="false"
          if [ -z "${associated_queues}" ];then
             associated_queues="${queue}"
          else
             associated_queues="${associated_queues},${queue}"
          fi
      fi
    else
       pdebug "false\n" 
    fi
done

#-- if any queue is currently associated with the virtual device, do not remove
if [ "${remove_device}" = "false" ];then
   if [ "${run_command_line}" = "true" ];then
      ${PRINT_CMD}
      ${PRINT_CMD} "The device [ ${virtual_device} ] is currently in use.  The following print queues"
      ${PRINT_CMD} "need to be associated with a different device before you can remove" 
      ${PRINT_CMD} "this device."
      ${PRINT_CMD}
      ${PRINT_CMD} "Printer Queues: ${associated_queues}"
   else
      ${PRINT_CMD} ${associated_queues}
   fi

   exit 2
else
   ${RM_CMD} -f "${VIR_DEVICES_DIR}/${virtual_device}"
   if [ "${?}" != "0" ];then
      ${PRINT_CMD} "Removing device [ ${virtual_device} ] was unsuccessful."
      exit 3 
   else
      ${PRINT_CMD} "Removing device [ ${virtual_device} ] was successful."
      exit 0
   fi
fi

exit 0
