#!/bin/sh

########################################################################
#             (c) Copyright 2003 Lexmark International, Inc.           #
#                        All rights reserved                           #
########################################################################
#
# This command creates printer queues.
#
########################################################################

#-- set debug to false by default. To turn debug "on" set to "1"
DEBUG=0

#-- initialize variables
run_command_line="true"


PRODUCT_LINK="/usr/genprint"
SCRIPTS_DIR="${PRODUCT_LINK}/bin/.scripts"

#-- load the shared script
. ${SCRIPTS_DIR}/shared

#-- initialize
initialize admin

#-- Get the correct modelfile based on app.properties
	if [ -f "${ROOT_SPOOL_DIR}/.${PRODUCT_NAME}/props/app.properties" ];then
		${GREP_CMD} "enable_asian=yes" ${ROOT_SPOOL_DIR}/.${PRODUCT_NAME}/props/app.properties >/dev/null 2>&1
		if [ "${?}" = "0" ];then
			MODEL_FILE="genprintmodela"
		else
			MODEL_FILE="genprintmodel"
		fi
	fi

#-- initialize classpath
set_classpath

program_name="mkqueue"

#-- Trap signal 2 and 15
trap 2 15

#-- Need special Java VM flags for languages, override the
#-- VMDEFINES variable. 
if [ -z "${VMDEFINES}" ];then
   if [ ! -z "${APPLICATION_LANG}" ];then
      LANGUAGE="`${PRINT_CMD} ${APPLICATION_LANG} | ${AWK_CMD} -F'_' '{print $1}'`"
      REGION="`${PRINT_CMD} ${APPLICATION_LANG} | ${AWK_CMD} -F'_' '{print $2}' | ${SED_CMD} -e 's/\..*$//'`"
      if [ "${REGION}" = "" ];then
         REGION="${LANGUAGE}"
      fi
      VMDEFINES="-Duser.language=${LANGUAGE} -Duser.region=${REGION} -Duser.locale=${APPLICATION_LANG}"
   fi
fi


#-- usage_short text
usage_short ()
{
${CAT_CMD} << EOT

USAGE:
    ${program_name} -q queue -d device -p type -l language [-o "opt1 opt2 ..."] ${SARG}
       [-x] 
    ${program_name} -h
    ${program_name} -v

Try '${program_name} -h' for more information

EOT
}


#-- usage_long text
usage_long ()
{

help_queue_name1="myT420"
help_prter_name1="Lexmark T420"
help_pdd_name1="10LT420"
help_queue_name2="myW820"
help_prter_name2="Lexmark W820"
help_pdd_name2="10LW820"

${CAT_CMD} << EOT

${program_name}
    Creates printer queues for devices created with the "mkdevice" command.

USAGE:
    ${program_name} -q queue -d device -p type -l language [-o "opt1 opt2 ..."] ${SARG}
       [-x]
    ${program_name} -h
    ${program_name} -v

OPTIONS:
    -q queue
       Specify a printer queue name.

    -d device
       Specify a virtual device name created by "mkdevice" to associate the new
       queue with.

    -p type
       Specify a printer model type.  A list of supported model types can be 
       found by executing "${PRODUCT_LINK}/bin/supported_printers"

    -l language
       Specify a printer language.  A list of supported languages can be found 
       by executing "${PRODUCT_LINK}/bin/supported_printers"
  
    -o "opt1 opt2 ..."
       Specify print queue options.  A list of supported print queue options
       can be found with the "lsqueue_opts" command.
    -x
       Run ${program_name} in debug mode.

    -h
       Display this help and exit.

    -v
       Display version information and exit.

EXAMPLES:
    Create a new printer queue, "myqueue", for a ${help_prter_name1} virtual 
    device, "${help_queue_name1}".  The language selection is specified as "automatic" 
    for this queue.  Specify additional printer options so that paper from 
    tray 2 should be used and that jobs should be duplexed against the long 
    edge of the paper:

       $ ${program_name} -q myqueue -d ${help_queue_name1} -p ${help_pdd_name1} -l automatic \\ 
               -o "paper_tray=tray2 duplex=long_edge"

    Create a new printer queue, "myqueue", for a ${help_prter_name2} virtual device,
    "${help_queue_name2}".  The language selection is specified as "pcl" for creating 
    a PCL emulation only queue.

       $ ${program_name} -q myqueue -d ${help_queue_name2} -p ${help_pdd_name2} -l pcl

EOT
}

#-- Check for Admin user
is_privileged_user
if [ "${?}" != "0" ];then
   ${PRINT_CMD}
   ${PRINT_CMD} "You must be an administrator to run this program."
   ${PRINT_CMD}
   exit 99
fi



#-- MAIN --#
#-- parse command line arguments --#
while getopts Ga:d:o:q:p:l:svhx arg
do
  case "${arg}" in
  G) run_command_line="false"
	  ASKSHUTDOWN="no"
     ;;
 
  d) device_name="${OPTARG}"
     ;;

  h) usage_long
     exit 99
     ;;

  q) queue_name="${OPTARG}"
     check_characters "${queue_name}" "Queue" || exit 99
     ;;

  o) printer_options="${OPTARG}"
     ;;

  p) printer_type="${OPTARG}"
     ;;

  l) printer_language="${OPTARG}"
     ;;

  s) ASKSHUTDOWN="no"
     ;;

  v) display_version
     exit 99
     ;;

  x) DEBUG=1
     ;;

  *) usage_short
     exit 99
     ;;
  esac
done

if   [ -z "${queue_name}" ];then
     usage_short
     ${PRINT_CMD} "Missing [-q] option, please specify a queue name."
     exit 99
elif [ -z "${device_name}" ];then
     usage_short
     ${PRINT_CMD} "Missing [-d] option, please specify a virtual device name."
     exit 99
fi

if [ "${run_command_line}" = "true" ];then
   if   [ -z "${printer_type}" ];then
        usage_short
        ${PRINT_CMD} "Missing [-p] option, please specify a printer type."
        exit 99
   elif [ -z "${printer_language}" ];then
        usage_short
        ${PRINT_CMD} "Missing [-l] option, please specify a printer language."
        exit 99
   fi
   verify_printer_options ${printer_type} ${printer_language}
   if [ $? -ne 0 ];then 
      exit 1
   fi
   if [ "${printer_language}" = "auto" ];then
      printer_language="automatic"
   fi
fi


pdebug "\nStep 1: Verify the existance of the virtual device.\n"
pdebug "---------------------------------------------------\n"

if [ ! -f "${VIR_DEVICES_DIR}/${device_name}" ];then
   pdebug "File : ${VIR_DEVICES_DIR}/${device_name}\n"
   ${PRINT_CMD} "Error: Virtual Device ${device_name} was not found."
   exit 1
fi

pdebug "Virtual Device is found.\n"
pdebug "..Continue..\n\n"


pdebug "\nStep 2: Verify the queue does not already exist.\n"
pdebug "---------------------------------------------------\n"

cqs_out=`check_queue_status ${queue_name} 2>&1`
RC=${?}
if [ "${RC}" = "0" ];then
   pdebug "Queue already exists.\n\n"
   ${PRINT_CMD} "Error: Print Queue [ ${queue_name} ] already exists."
   exit 1
elif [ "${RC}" = "2" ];then
   pdebug "Printer Queue or class name already exists, exit 1\n\n"
   ${PRINT_CMD} "Error: Printer queue or class name [ ${queue_name} ] already exists."
   exit 2
fi

if [ "${DEBUG}" = "1" ];then
   ${PRINTF_CMD} "Check Queue Status Output\n" >&2
   ${PRINTF_CMD} "-------------------------\n" >&2
   ${PRINTF_CMD} "${cqs_out}\n" >&2
   ${PRINTF_CMD} "-------------------------\n" >&2
fi

pdebug "Printer queue [ ${queue_name} ] does not exist.\n"
pdebug "..Continue..\n\n"


if [ "${run_command_line}" = "true" ];then
   pdebug "Now build virtual queue definition\n"

   #-- CLASSPATH is set in shared
   TARGET="com.lexmark.unix.drivers.jprint.printeroptions.MakeQueue"

   LANG=${APPLICATION_LANG}
   export LANG
   CMD="${JAVA_CMD} ${VMDEFINES} -cp ${CLASSPATH} ${TARGET} ${queue_name} desc ${printer_type} ${device_name} ${printer_language} ${printer_options}"
   pdebug "Running:\n${CMD}\n"
   if [ "${DEBUG}" = "1" ];then
      ${CMD}
   else
      ${CMD} >/dev/null 2>&1
   fi
   if [ "${?}" != "0" ]; then
      ${PRINT_CMD} "Creating printer queue ${queue_name} failed."
      exit 1
   fi
   pdebug "Creating the virtual queue definition was successful.\n"
   LANG=C
   export LANG
fi


pdebug "\nStep 3: Create the queue\n"
pdebug "---------------------------------------------------\n"


${LPADMIN_CMD} -p ${queue_name} -i ${PRODUCT_LINK}/etc/model/${MODEL_FILE}
if [ "${?}" != "0" ];then 
   pdebug "Creating queue failed.., exit 1\n"
   ${PRINT_CMD} "Error: Creating printer queue ${queue_name} failed."
   pdebug "Remove the virtual queue.\n" 
   cd ${VIR_QUEUES_DIR}
   ${RM_CMD} -rf ${queue_name} >/dev/null 2>&1
   exit 1
fi 
 
pdebug "Queue created successfully\n"
pdebug "..Continue ..\n\n"


pdebug "\nStep 4: Set queue to Enabled and Accepting\n"  
pdebug "---------------------------------------------------\n"
AOUT=`${ACCEPT_CMD} ${queue_name} 2>&1`
if [ "${?}" != "0" ];then
	#-- I added a sleep for Solaris 10.  If the system does not have
	#-- a queue defined, lpsched will not be running.  lpadmin, starts
	#-- the lpsched, however it takes a second or so, accept will fail
	#-- if we call it to early.
	sleep 3
	${ACCEPT_CMD} ${queue_name}
	if [ "${?}" != "0" ];then
		pdebug "accept ${queue_name} failed, exit 1\n"
		${PRINT_CMD} "Creating printer queue ${queue_name} was successful, however"
		${PRINT_CMD} "accepting print requests for printer queue ${1} has failed."
		exit 1
	fi
else
	if [ "${AOUT}" != "" ];then
		${PRINT_CMD} ${AOUT}
	fi
fi
pdebug "Accepting print requests complete .\n"
${ENABLE_CMD} ${queue_name}
if [ "${?}" != "0" ];then
   pdebug "enable ${queue_name} failed, exit 1\n"
   ${PRINT_CMD} "Creating printer queue ${queue_name} was successful, however"
   ${PRINT_CMD} "enabling printer queue ${queue_name} was unsuccessful."
   exit 1
fi
pdebug "Accepting print requests complete .\n\n"

${PRINT_CMD} "Queue ${queue_name} was created successfully."
exit 0;

